# -*- coding: utf-8 -*-

import logging
import os

import yaml

import sandbox.projects.inventori.common.resources as resources
import sandbox.sdk2 as sdk2
from enum import Enum
from sandbox.projects.common import binary_task
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.projects.inventori.common import InventoryRunTaskTemplate

LOGGER_NAME = "inventori performance etl generated"
logger = logging.getLogger(LOGGER_NAME)
SOLOMON_PROJECT = 'inventori'
SOLOMON_SERVICE = 'scheduler'

SANDBOX_RESOURCE_NAME = "yt-data"
MINIMAL_ARCHIVE_WEIGHT_KBYTES = 1  # kilo bytes


def send_email_and_deploy(self, prev_status):
    def convert_ttl(ttl_param):
        ttl_param = int(ttl_param)
        return 'inf' if ttl_param == 0 else ttl_param

    def choose_type_to_release(env_type):
        if env_type == 'prod':
            return 'stable'
        elif env_type == 'test':
            return 'testing'
        elif env_type == 'dev':
            return 'unstable'

    releaseTo = choose_type_to_release(self.task_params.environment_type)
    self.set_info('Will release database to {releaseTo}'.format(releaseTo=releaseTo))

    if self.Context.has_new_resource and releaseTo and self.Parameters.release_new_resource:
        additional_parameters = dict(
            releaser=self.author,
            release_status=releaseTo,
            release_subject='Automatic release of ' + self.Parameters.description,
            email_notifications=dict(to=['inventori-monitoring@yandex-team.ru'], cc=[]),
        )
        if isinstance(self, release_integration.ReleaseToYaDeployTask2):
            release_integration.ReleaseToYaDeployTask2.on_release(self, additional_parameters)
        if isinstance(self, sdk2.Task):
            self.mark_released_resources(
                releaseTo,
                ttl=convert_ttl(self.Parameters.ttl),
            )

    sdk2.Task.on_success(self, prev_status)


class CONFIG_PATHS(str, Enum):  # magic inheritance ;)
    DEV_CONFIG = "dev_config"
    TEST_CONFIG = "test_config"
    PROD_CONFIG = "prod_config"


def get_config_parsed(config_name):
    """Return parsed config in format {'//home/inventori/recommendation1' : 'cid, uid, ...'}"""
    try:
        from library.python import resource
    except ImportError:
        return {}
    else:
        source_config = yaml.safe_load(resource.find("/{config_name}".format(config_name=config_name)))

        logger.info('source_config is: {config}'.format(config=source_config))
        config_parsed = {}

        source_config = {table: source_config[table] for table in source_config if
                         'table_fields' in source_config[table]}

        for table_name in source_config:
            # merging
            values = source_config[table_name]

            if 'basic_table_fields' in values:
                values['table_fields'] += ', ' + values['basic_table_fields']
                del values['basic_table_fields']

            if 'autobudget_table_fields' in values:
                values['table_fields'] += ', ' + values['autobudget_table_fields']
                del values['autobudget_table_fields']

            config_parsed[values['path']] = values['table_fields']

        return config_parsed


def pretty_config_output(d, indent=0, result=""):
    for key, value in d.items():
        result += '\t' * indent + str(key) + '\n'
        if isinstance(value, dict):
            result += pretty_config_output(value, indent + 1) + '\n'
        else:
            result += '\t' * (indent + 1) + str(value) + '\n'

    return result


class ReleaseParams(sdk2.Parameters):
    ext_params = binary_task.binary_release_parameters(custom=True)


class InventoriPerformanceEtl(
    binary_task.LastBinaryTaskRelease,
    release_integration.ReleaseToYaDeployTask2,
    InventoryRunTaskTemplate.InventoryTaskParamsTemplate,
    sdk2.Task,
):
    class Context(sdk2.Task.Context):
        has_new_resource = False

    class Requirements(sdk2.Requirements):
        disk_space = 10 * 1024  # 10 Gb for data (all necessary indices + archive)
        cores = 1

    class Parameters(
        InventoryRunTaskTemplate.get_run_params(
            resources.InventoriPerformanceEtlBinary.InventoriTaskParams,
            base_class=ReleaseParams,
        )
    ):
        ttl = sdk2.parameters.Integer("TTL for released resource (days, always; 0 for inf)",
                                      default=30, required=True)

        # release_new_resource, releaseTo = old_utils.releaseTo_params()
        release_new_resource = sdk2.parameters.Bool(
            "Release new resource when it is created.",
            default=False,
            required=True,
        )

    def get_yp_oauth_token(self):
        return sdk2.Vault.data("INVENTORI_YP_TOKEN")

    def on_execute(self):
        from inventori.pylibs.utils.yt_tables_utils import read_several_tables_and_create_archive
        from datetime import datetime
        import copy

        self.Context.start_time = int(datetime.now().timestamp())
        self.Context.save()

        binary_task.LastBinaryTaskRelease.on_execute(self)
        from inventori.pylibs.tasks.etl import performance_etl_collector

        if not hasattr(self.task_params, 'task_name') or not self.task_params.task_name:
            self.task_params.task_name = resources.InventoriPerformanceEtlBinary.task_name

        self._send_metric_to_solomon('status', 0)

        # We are sending statuses and duration from sb-runner =>
        # don't want product task to send the same data
        product_task_params = copy.deepcopy(self.task_params)
        product_task_params.solomon_token = None

        if self.task_params.environment_type == 'prod':
            parsed_config = get_config_parsed(CONFIG_PATHS.PROD_CONFIG.value)
        elif self.task_params.environment_type == 'test':
            logging.info('in test')
            parsed_config = get_config_parsed(CONFIG_PATHS.TEST_CONFIG.value)
        else:
            logging.info('in dev')
            parsed_config = get_config_parsed(CONFIG_PATHS.DEV_CONFIG.value)

        logger.info('Current config is {config}'.format(config=pretty_config_output(parsed_config)))
        self.set_info(pretty_config_output(parsed_config))

        # running product code
        task = performance_etl_collector.PerformanceEtlCollector(product_task_params, parsed_config)
        task.run()

        self.set_info(
            "Here backup of current recommendations: <a href=https://yt.yandex-team.ru/hahn/navigation?path={0}> "
            "{0} </a> ".format(self.task_params.yt_database_dir),
            do_escape=False,
        )

        # archiving tables and making resource
        archive_created = read_several_tables_and_create_archive(
            logger=logger,
            log_processor=lambda: sdk2.helpers.ProcessLog(self, logger=LOGGER_NAME),
            yt_proxy=self.task_params.yt_proxy,
            yt_token=self.task_params.yt_token,
            # yt_tables = parsed_config
            sandbox_resource_name=SANDBOX_RESOURCE_NAME,
            yt_tables=parsed_config
        )

        if archive_created:
            archive_name = '{archive_name}.{extension}'.format(archive_name=SANDBOX_RESOURCE_NAME, extension='tar.gz')
            size = os.path.getsize(archive_name)
            size_kb = size >> 10
            self.set_info('Archive {archive_name} was created, size is: {size_kb} KB'.format(archive_name=archive_name,
                                                                                             size_kb=size_kb))
            if size_kb < MINIMAL_ARCHIVE_WEIGHT_KBYTES:
                self.set_info(
                    "{emoji} Archive must be empty, I won't create any resource to deploy {emoji}".format(emoji='😡'))
                raise ValueError('EMPTY archive')

            self._send_metric_to_solomon('archive_size(KB)', size_kb)

            made_resource = sdk2.ResourceData(
                resources.InventoriPerformanceDatabase(
                    self,
                    "Inventori performance database in tar.gz",
                    archive_name
                )
            )
            made_resource.ready()
            self.Context.has_new_resource = True

            self.set_info("Create resource {resource}".format(resource=made_resource))
        else:
            raise ValueError("Can't create archive due to some reasons")

        self._send_metric_to_solomon('status', 1)

    def _send_metric_to_solomon(self, metric_name, value):
        if not self.Parameters.send_status_to_solomon:
            # do nothing if ask do NOT send to solomon
            return

        from datetime import datetime
        from inventori.pylibs.solomon import Solomon

        solomon_client = Solomon(
            oauth_token=self.task_params.solomon_token,
            project=SOLOMON_PROJECT,
            cluster=self.task_params.environment_type,
            service=SOLOMON_SERVICE,
        )

        payload = [{
            'ts': self.Context.start_time,
            'labels': {
                'yt_cluster': self.task_params.yt_yql_cluster,
                'metric': metric_name,
                'task_name': str(self.task_params.task_name),
            },
            'value': value,
        }]

        logger.info('Sending %s {metric_name} to solomon:\n{payload}'.format(metric_name=metric_name, payload=payload))
        solomon_client.send(payload)

        if metric_name == 'status' and value != 0:
            logging.info('Also send task duration')

            payload = [{
                'ts': self.Context.start_time,
                'labels': {
                    'yt_cluster': self.task_params.yt_yql_cluster,
                    'metric': 'duration',
                    'task_name': str(self.task_params.task_name),
                },
                'value': int(datetime.now().timestamp()) - self.Context.start_time,
            }]

            logger.info(('Sending %s duration to solomon:\n%s', payload))
            solomon_client.send(payload)

    def on_success(self, prev_status):
        send_email_and_deploy(self, prev_status)

    def on_break(self, prev_status, status):
        logging.info('On break, prev_status=$s, status=%s', prev_status, status)
        self._send_metric_to_solomon('status', -1)

        super().on_break(prev_status, status)
