# -*- coding: utf-8 -*-
import logging
from sandbox.projects.common.ya_deploy import release_integration
import sandbox.projects.common.platform_api.platform_api as platform_api
import sandbox.projects.inventori.common.consts as consts
import sandbox.sandboxsdk as sandbox_sdk
import sandbox.sdk2 as sdk2
import shutil
import subprocess


def convert_ttl(ttl_param):
    """
    Convert user input to ttl.

    >>> convert_ttl(0)
    'inf'
    >>> convert_ttl(1)
    1
    >>> convert_ttl(5)
    5

    :param ttl_param: ttl from user input
    :return: 'inf' if ttl is 0, else ttl itself.
    """
    ttl_param = int(ttl_param)  # иногда приходит float
    return 'inf' if ttl_param == 0 else ttl_param


def releaseTo_selector(title, with_empty=False):
    """Radio button for releasing branch selection"""
    with sdk2.parameters.RadioGroup(title) as releaseTo:
        if with_empty:
            assert 'any' not in sandbox_sdk.sandboxapi.RELEASE_STATUSES
            releaseTo.values['any'] = releaseTo.Value(value='any', default=True)
        for val in sandbox_sdk.sandboxapi.RELEASE_STATUSES:
            releaseTo.values[val] = releaseTo.Value(value=val)
    return releaseTo


def releaseTo_params():
    """
    Usage:

    class MyTask(sdk2.Task):
        class Parameters(sdk2.Task.Parameters):
            release_new_resource, releaseTo = releaseTo_params()
    """
    release_new_resource = sdk2.parameters.Bool(
        "Release new resource when it is created.",
        default=False,
        required=True,
    )

    with release_new_resource.value[True]:
        releaseTo = releaseTo_selector(title="Automatically release resource to", with_empty=False)

    return release_new_resource, releaseTo


def read_and_save_yt_tables(format, client, logger, tables_prefixes_to_columns, tables_list, yt_data_folder):
    import yt.wrapper as yt
    for table_name_or_prefix_mask in tables_prefixes_to_columns:
        logger.info("-- processing %s" % table_name_or_prefix_mask)
        if table_name_or_prefix_mask.endswith('*'):
            table_prefix = table_name_or_prefix_mask.rstrip('*')
            current_tables_list = list(filter(lambda table_name: table_name.startswith(table_prefix), tables_list))
        else:
            current_tables_list = list(filter(lambda table_name: table_name == table_name_or_prefix_mask, tables_list))
        logger.info("-- processing %s tables: [%s]", table_name_or_prefix_mask, ','.join(current_tables_list))

        for table in current_tables_list:
            logger.info("-- loading table prefix - %s table - %s", table_name_or_prefix_mask, table)

            columns = tables_prefixes_to_columns.get(table_name_or_prefix_mask)

            if format == consts.JSON_FORMAT:
                table_to_read = '%s/%s{%s}' % (yt_data_folder, table, columns)
                format_to_read = yt.JsonFormat()
                file_name = '%s/%s.json' % (consts.PLATFORM_YT_DATA_FOLDER, table)
            elif format == consts.CSV_FORMAT:
                table_to_read = '%s/%s' % (yt_data_folder, table)
                format_to_read = '<columns=[%s];field_separator=";";missing_value_mode=print_sentinel>schemaful_dsv' % columns
                file_name = '%s/%s' % (consts.PLATFORM_YT_DATA_FOLDER, table)
            else:
                raise Exception('unknown format %s' % format)

            logger.info("-- loading table by path - %s to file - %s", table_to_read, file_name)

            file = open(file_name, 'wb')
            yt_table_iterator = client.read_table(table_to_read, format=format_to_read, raw=True)
            shutil.copyfileobj(yt_table_iterator, file)
            file.close()
            logger.info("-- loading of table prefix - %s table - %s finished", table_name_or_prefix_mask, table)


def read_tables_and_create_archive(log_processor, logger, yt_proxy, yt_token, yt_data_folder,
                                   csv_tables_prefix_to_columns=(),
                                   json_tables_prefix_to_columns=()):
    import yt.wrapper as yt

    client = yt.YtClient(proxy=yt_proxy, token=yt_token)

    archive_created = False

    if client.exists(yt_data_folder):

        tables_list = client.list(yt_data_folder)

        if tables_list:
            with log_processor() as process_log:
                subprocess.check_call('mkdir %s' % consts.PLATFORM_YT_DATA_FOLDER, shell=True,
                                      stderr=process_log.stdout,
                                      stdout=process_log.stdout)

            #   reading files in csv format
            logger.info("-- reading files in csv format: " + str(csv_tables_prefix_to_columns))
            read_and_save_yt_tables(format=consts.CSV_FORMAT,
                                    client=client,
                                    logger=logger,
                                    tables_prefixes_to_columns=csv_tables_prefix_to_columns,
                                    tables_list=tables_list,
                                    yt_data_folder=yt_data_folder)

            #   reading files in json format
            logger.info("-- reading files in json format: " + str(json_tables_prefix_to_columns))
            read_and_save_yt_tables(format=consts.JSON_FORMAT,
                                    client=client,
                                    logger=logger,
                                    tables_prefixes_to_columns=json_tables_prefix_to_columns,
                                    tables_list=tables_list,
                                    yt_data_folder=yt_data_folder)

            #
            #   archiving files
            #

            with log_processor() as process_log:
                subprocess.check_call(
                    'cd %s && tar -czvf ../%s ./*' % (consts.PLATFORM_YT_DATA_FOLDER, consts.PLATFORM_YT_ARCHIVE_NAME),
                    shell=True, stderr=process_log.stdout, stdout=process_log.stdout)

            archive_created = True

    return archive_created


def send_email_and_deploy(self, prev_status):
    if self.Context.has_new_resource and self.Parameters.releaseTo and self.Parameters.release_new_resource:
        additional_parameters = dict(
            releaser=self.author,
            release_status=self.Parameters.releaseTo,
            release_subject='Automatic release of ' + self.Parameters.description,
            email_notifications=dict(to=['inventori-monitoring@yandex-team.ru'], cc=[]),
        )
        if isinstance(self, release_integration.ReleaseToYaDeployTask2):
            release_integration.ReleaseToYaDeployTask2.on_release(self, additional_parameters)
        if isinstance(self, sdk2.Task):
            self.mark_released_resources(
                self.Parameters.releaseTo,
                ttl=convert_ttl(self.Parameters.ttl),
            )

    sdk2.Task.on_success(self, prev_status)
