# -*- coding: utf-8 -*-
import os

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.inventori.InventoriArcYaMake2 import InventoriArcYaMake2
from sandbox.projects.inventori.common.InventoryRunTaskTemplate import get_resource_task_name
from sandbox.projects.inventori.common.resources import INVENTORI_TASK_BINARIES_RESOURCES, ROBOT_INVENTORI_SECRET
from sandbox.projects.inventori.common.utils import cached_property
import sandbox.projects.sandbox.deploy_binary_task as dbt
import logging


def get_project_params(resources, base_class=sdk2.Task.Parameters):
    class Parameters(base_class):
        resources_to_build = resources
        multiple_targets = sdk2.parameters.Bool('Multiple targets')
        with multiple_targets.value[True]:
            with sdk2.parameters.CheckGroup('Resources to build') as target_resources:
                for res in resources:
                    setattr(target_resources.values, str(res), target_resources.Value(str(res), checked=False))
        with multiple_targets.value[False]:
            with sdk2.parameters.RadioGroup('Resource to build') as target_resource:
                for res in resources:
                    setattr(target_resource.values, str(res), target_resource.Value(str(res)))

    return Parameters


class BuildInventoriTasks(InventoriArcYaMake2):
    class Parameters(get_project_params(INVENTORI_TASK_BINARIES_RESOURCES,
                                        base_class=InventoriArcYaMake2.Parameters)):
        ext_params = binary_task.binary_release_parameters_list(none=True)

        ya_make_project_params = None  # remove this params as we build specific target(s)

        build_runner = sdk2.parameters.Bool('Wanna rebuild RunInventoriTasks',
                                            description='If you are adding new task type or changing tasks '
                                                        'InventoriTaskParams - you should also build and release task '
                                                        'runner',
                                            default=True, required=True)

        with build_runner.value[True]:
            with sdk2.parameters.Group("Runner build options",
                                       ) as runner_build_options:
                target_for_runner_task = sdk2.parameters.String(
                    'Path to Runner task relative to Arcadia root',
                    default='sandbox/projects/inventori/tasks/RunInventoriTasks/bin/run-inventori-task',
                )

                runner_task_type = sdk2.parameters.String(
                    'Name of the runner task type to tag and check for',
                    default='RUN_INVENTORI_TASKS',
                )

                release_ttl = sdk2.parameters.Integer('TTL for released resource (days, always; 0 for inf)',
                                                      default=60, required=True)

                arc_oauth_token = sdk2.parameters.YavSecret(
                    "Arc token vault entry ",
                    required=True,
                    default="{robot_secret}#{arc_token}".format(robot_secret=ROBOT_INVENTORI_SECRET,
                                                                arc_token='arc_token')
                )

    @cached_property
    def target_resource_classes(self):
        resources = []
        if self.Parameters.multiple_targets:
            target_resources = self.Parameters.target_resources
        else:
            target_resources = {self.Parameters.target_resource}
        for resource in self.Parameters.resources_to_build:
            if resource in target_resources:
                resources.append(resource)
        return resources

    def get_resources(self):
        resources = {}
        for resource in self.target_resource_classes:
            resources.update({
                str(resource): {
                    'description': getattr(
                        resource,
                        'task_executable_resource_description',
                        'Build {} executable'.format(
                            os.path.basename(resource.arcadia_build_path))),
                    'resource_path': os.path.join(
                        str(resource), os.path.basename(resource.arcadia_build_path)),
                    'resource_type': resource,
                }
            })
        return resources

    def get_targets(self):
        return [os.path.dirname(resource.arcadia_build_path) for resource in self.target_resource_classes]

    def get_arts(self):
        return [
            {
                'path': resource.arcadia_build_path.rstrip('/'),  # где будет лежать артефакт после сборки
                'dest': str(resource),  # каталог, куда поместить артефакт относительно корня архива-ресурса
            }
            for resource in self.target_resource_classes
        ]

    def on_enqueue(self):
        if len(self.target_resource_classes) == 0:
            raise Exception('No resources to build')
        for resource in self.target_resource_classes:
            eh.ensure(
                hasattr(resource, 'arcadia_build_path'),
                'Resource {} does not have "arcadia_build_path" attribute'.format(str(resource))
            )
        super(BuildInventoriTasks, self).on_enqueue()

    def on_execute(self):
        logging.info('build from {}'.format(self.Parameters.checkout_arcadia_from_url))

        for resource in self.target_resource_classes:
            tag = get_resource_task_name(resource)
            if tag not in self.Parameters.tags:
                self.Parameters.tags.append(tag)

        # Build runner
        if self.Parameters.build_runner:
            dbt_task = dbt.DeployBinaryTask(
                self,
                arcadia_url=self.Parameters.checkout_arcadia_from_url,
                arcadia_patch=self.Parameters.arcadia_patch,
                description='Building new binary for {} task'.format(self.Parameters.runner_task_type),
                target=self.Parameters.target_for_runner_task,
                attrs={
                    'task_type': self.Parameters.runner_task_type,
                    'taskbox_enabled': True
                },
                check_types=[self.Parameters.runner_task_type],
                use_yt_cache=False,
                use_arc=True,
                arc_oauth_token=self.Parameters.arc_oauth_token,
                release_ttl=self.Parameters.release_ttl,
                binary_executor_release_type=self.Parameters.binary_executor_release_type,
            )
            dbt_task.enqueue()

        # Build Chosen "product" binary task
        super(BuildInventoriTasks, self).on_execute()

        if self.Parameters.build_runner:  # TODO: should wait for db_task
            self.set_info(
                "Built runner tasks <a href=https://sandbox.yandex-team.ru/task/{0}> https://sandbox.yandex-team.ru/task/{0} </a> <b>Do not forget to release "
                "it!</b>".format(dbt_task.id),
                do_escape=False,
            )

        if isinstance(self.Parameters.target_resource, str) and '_' in self.Parameters.target_resource :
            tag = self._get_tags_name_from_parameters(self.Parameters.target_resource)

            affected_schedulers_link = "https://sandbox.yandex-team.ru/schedulers?all_tags=false" \
                              "&owner=INVENTORI&tags={tag1}%2C{tag2}&task_type=RUN_INVENTORI_TASKS".format(tag1=tag,
                                                                                                           tag2=self.Parameters.target_resource)
            self.set_info(
                "Scheduler u must want to change <a href={0}> {1} schedulers </a> 👀".format(affected_schedulers_link, tag),
                do_escape=False,
            )

    def _get_tags_name_from_parameters(self, resource_name):
        s_spl = resource_name.split('_')
        return "_".join(s_spl[s_spl[0].startswith("INVEN"):-(s_spl[-1] == 'BINARIES')])
