# -*- coding: utf-8 -*-
import datetime
import logging

import requests

import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sdk2
from sandbox.sandboxsdk.channel import channel

logger = logging.getLogger(__name__)


class IOTRegressTests(sdk2.Task):
    MAD_HATTER_GROUP = 'WONDERLAND'
    MAD_HATTER_ST_TOKEN = 'env.STARTREK_OAUTH_TOKEN'
    MAD_HATTER_TESTPALM_TOKEN = 'env.TESTPALM_OAUTH_TOKEN'

    class Parameters(sdk2.Task.Parameters):
        release_ticket_id = sdk2.parameters.String('Release ticket key', default="ALICERELEASE-XXX", required=True)
        assessors_queue = sdk2.parameters.String('Assessors queue', default="ALICEASSESSORS", required=True)
        alice_queue = sdk2.parameters.String('Alice queue', default="ALICE", required=True)
        project_id = sdk2.parameters.String('Testpalm project', default="iot-ui", required=True)
        tag = sdk2.parameters.String('Regress testsuite tag', default="smarthome", required=True)

    class Requirements(sdk2.Requirements):
        client_tags = task_env.TaskTags.all_rm & task_env.TaskTags.startrek_client
        environments = [
            task_env.TaskRequirements.startrek_client,
        ]

    def on_execute(self):
        release_ticket_id = self.Parameters.release_ticket_id
        assessors_queue = self.Parameters.assessors_queue
        alice_queue = self.Parameters.alice_queue
        project_id = self.Parameters.project_id
        tag = self.Parameters.tag

        st_token, testpalm_token = self.get_tokens()
        testpalm_client = TestpalmClient(testpalm_token)
        created_runs = testpalm_client.create_runs(project_id, tag, release_ticket_id, assignees=[])

        st_client = StartrekClient(st_token, useragent="alice-iot")
        # st_client.link_regress_runs(created_runs, release_ticket_id)
        st_client.create_ticket(assessors_queue, created_runs, release_ticket_id)
        st_client.create_ticket(alice_queue, [], release_ticket_id)

    def get_tokens(self):
        logger.info(
            "Receiving st {st} and testpalm {testpalm} tokens of group {group}".format(
                st=self.MAD_HATTER_ST_TOKEN,
                testpalm=self.MAD_HATTER_TESTPALM_TOKEN,
                group=self.MAD_HATTER_GROUP,
            )
        )

        try:
            st_token = channel.task.get_vault_data(self.MAD_HATTER_GROUP, self.MAD_HATTER_ST_TOKEN)
            testpalm_token = channel.task.get_vault_data(self.MAD_HATTER_GROUP, self.MAD_HATTER_TESTPALM_TOKEN)
        except Exception as error:
            raise Exception("Unable to get tokens, reason: {error}".format(error=error))

        logger.info(
            "Tokens received successfully"
        )
        return st_token, testpalm_token


class TestpalmClient(object):
    TESTPALM_API_URL = "https://testpalm-api.yandex-team.ru"
    TESTPALM_URL = "https://testpalm.yandex-team.ru"

    def __init__(self, testpalm_token):
        self.testpalm_token = testpalm_token

    @staticmethod
    def generate_headers(token):
        return {
            'Authorization': 'OAuth {}'.format(token),
            'Content-Type': 'application/json',
        }

    def create_runs(self, project_id, tag, ticket_id, assignees):
        logger.info(
            "Creating runs in project {project_id} for ticket {ticket_id} and tag {tag}".format(
                project_id=project_id,
                ticket_id=ticket_id,
                tag=tag
            )
        )

        try:
            suites = self.get_suites_by_tag(project_id, tag)
        except Exception as error:
            raise Exception("Unable to create regress runs, reason: {error}".format(error=error))
        created_runs = []
        for suite in suites:
            try:
                title = self.generate_run_title(suite.get("title"), ticket_id, tag, datetime.datetime.now())
                run = self.create_run(project_id, title, assignees, suite, ticket_id, suite.get("tags"))
                created_runs.append(run)
            except Exception as error:
                raise Exception("Unable to create regress run, reason: {error}".format(error=error))

        logger.info(
            "Created runs {created_runs}".format(
                created_runs=created_runs
            )
        )
        return created_runs

    def get_suites_by_tag(self, project_id, tag):
        logger.info(
            "Getting suites in project {project_id} for tag {tag}".format(
                project_id=project_id,
                tag=tag
            )
        )

        url = "{base_url}/testsuite/{project_id}".format(
            base_url=self.TESTPALM_API_URL,
            project_id=project_id,
        )
        headers = self.generate_headers(self.testpalm_token)
        params = {
            "include": "id,title,tags",
            "expression": '''{{"type": "CONTAIN", "key": "tags", "value": "{tag}" }}'''.format(tag=tag)
        }
        response = requests.get(url, headers=headers, params=params)
        if response.status_code != 200:
            raise Exception(
                "Unable to get testpalm suites response, code: {code}, body: {body}".format(
                    code=response.status_code,
                    body=response.text,
                )
            )
        suites = response.json()

        logger.info(
            "Received suites {suites}".format(
                suites=suites
            )
        )
        return suites

    def create_run(self, project_id, title, assignees, suite, ticket_id, testpalm_tags):
        logger.info(
            "\tCreating run {title} in project {project_id} for suite {suite_id}".format(
                title=title,
                project_id=project_id,
                suite_id=suite.get("id")
            )
        )

        url = "{base_url}/testrun/{project_id}/create".format(
            base_url=self.TESTPALM_API_URL,
            project_id=project_id,
        )
        headers = self.generate_headers(self.testpalm_token)
        params = {
            "include": "id,title",
        }
        payload = {
            "tags": testpalm_tags,
            "title": title,
            "testSuite": {
                "id": suite.get("id")
            },
            "assignee": assignees,
            "ignoreSuiteOrder": False,
            "parentIssue": {
                "id": ticket_id,
                "trackerId": "Startrek"
            }
        }
        response = requests.post(url, json=payload, params=params, headers=headers)
        if response.status_code != 200:
            raise Exception(
                "Unable to create testpalm run, code: {code}, body: {body}".format(
                    code=response.status_code,
                    body=response.text,
                )
            )
        run = next(iter(response.json()), {})
        run["url"] = "{base_url}/{project_id}/testrun/{id}".format(
            base_url=self.TESTPALM_URL,
            project_id=project_id,
            id=run.get('id')
        )
        run["key"] = "testrun/{project_id}/{id}".format(
            project_id=project_id,
            id=run.get('id')
        )
        run["tags"] = suite.get("tags")

        logger.info(
            "\tCreated run {run}".format(
                run=run
            )
        )
        return run

    @staticmethod
    def generate_run_title(title, ticket_id, tag, date):
        title = "{title}: {ticket_id} {tag} {date}".format(
            title=title,
            ticket_id=ticket_id,
            tag=tag,
            date=date.strftime("%d.%m"),
        )
        return title


class StartrekClient(object):
    def __init__(self, st_token, useragent=None):
        if not useragent:
            useragent = "alice-iot"
        from startrek_client import Startrek
        self.client = Startrek(useragent=useragent, token=st_token)

    @staticmethod
    def generate_headers(token):
        return {
            'Authorization': 'OAuth {}'.format(token),
            'Content-Type': 'application/json',
        }

    def create_ticket(self, queue, created_runs, release_ticket_id):
        logger.info(
            "Creating ticket in queue {queue} for release ticket {release_ticket_id}".format(
                queue=queue,
                release_ticket_id=release_ticket_id
            )
        )

        if created_runs:
            title = self.assessors_title(release_ticket_id)
            description = self.assessors_description(release_ticket_id, created_runs)
            try:
                ticket = self.client.issues.create(
                    queue=queue,
                    summary=title,
                    description=description,
                    type='task'
                )
            except Exception as error:
                raise Exception("Unable to create ticket, reason: {error}".format(error=error))
        else:
            title = self.alice_title(release_ticket_id)
            description = self.alice_description(release_ticket_id)
            try:
                ticket = self.client.issues.create(
                    queue=queue,
                    summary=title,
                    description=description,
                    assignee={'id': 'levkovskayama'},
                    tags='cases',
                    type='task'
                )
            except Exception as error:
                raise Exception("Unable to create ticket, reason: {error}".format(error=error))

        logger.info(
            "Created ticket {ticket}".format(
                ticket=ticket
            )
        )
        return ticket

    def link_regress_runs(self, created_runs, release_ticket_id):
        logger.info(
            "Linking regress runs to release ticket {release_ticket_id}".format(
                release_ticket_id=release_ticket_id
            )
        )

        try:
            release_ticket = self.client.issues[release_ticket_id]
        except Exception as error:
            raise Exception("Unable to get release ticket, reason: {error}".format(error=error))
        for run in created_runs:
            try:
                self.link_regress_run(release_ticket, run)
            except Exception as error:
                raise Exception("Unable to link regress runs, reason: {error}".format(error=error))

        logger.info(
            "All regress runs linked successfully"
        )

    @staticmethod
    def link_regress_run(release_ticket, run):
        logger.info(
            "\tLinking run {run}".format(
                run=run
            )
        )

        run_key = run.get("key")
        try:
            release_ticket.remotelinks.create(origin="ru.yandex.testpalm", key=run_key, relationship='relates')
        except Exception as error:
            raise Exception(
                "Unable to link regress run {run_key} to release ticket, reason: {error}".format(
                    run_key=run_key,
                    error=error
                )
            )

        logger.info(
            "\tRun {run_key} linked successfully".format(
                run_key=run_key
            )
        )

    @staticmethod
    def assessors_title(release_ticket_id):
        return "Тестирование релиза smarthome {release_ticket_id}".format(
            release_ticket_id=release_ticket_id
        )

    @staticmethod
    def alice_title(release_ticket_id):
        return "Кейсы по релизу {release_ticket_id}".format(
            release_ticket_id=release_ticket_id
        )

    @staticmethod
    def generate_run_line(run):
        return "[{tags}] {url}".format(tags=" ".join(run.get("tags")), url=run.get("url"))

    @staticmethod
    def alice_description(release_ticket_id):
        header = 'Тикет {release_ticket_id}'.format(release_ticket_id=release_ticket_id)
        return (
            '{header}\n'
        ).format(header=header)

    @staticmethod
    def assessors_description(release_ticket_id, created_runs):
        header = (
            '{release_ticket_id}'
        ).format(release_ticket_id=release_ticket_id)

        release_stand_configuration = (
            '====Конфигурация для тестирования====\n'
            'Тестран для мобильных проходим **в ПУДЯ с дебаг панелью**, залипнув в тестид 582877 и проставив VINS и Uniproxy'  # noqa
            '#|\n'
            '|| **Uniproxy URL** | wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-hamster/uni.ws?srcrwr=IOT_USER_INFO:VOICE__IOT_USER_INFO_BETA ||\n'  # noqa
            '|| **Vins URL** | http://vins.hamster.alice.yandex.net/speechkit/app/pa/?srcrwr=IoT:iot-beta.quasar.yandex.net:80&srcrwr=SCENARIO_IOT_APPLY:iot-beta.quasar.yandex.net:80&srcrwr=IoTScenarios:iot-beta.quasar.yandex.net:80&srcrwr=SCENARIO_IOTSCENARIOS_APPLY:iot-beta.quasar.yandex.net:80&srcrwr=IoTVoiceDiscovery:iot-beta.quasar.yandex.net:80&srcrwr=SCENARIO_IOTVOICEDISCOVERY_APPLY:iot-beta.quasar.yandex.net:80 ||\n'  # noqa
            '|| **Ссылка на UI (iot) в браузере** | https://hamster.yandex.ru/iot?srcrwr=IOT_HOST%3Aiot-beta.quasar.yandex.ru&srcrwr=IOT__USER_INFO:SHARED__IOT_USER_INFO_BETA ||\n'  # noqa
            '|| **Ссылка на UI (quasar) в браузере** | https://hamster.yandex.ru/quasar?srcrwr=IOT_HOST%3Aiot-beta.quasar.yandex.ru&srcrwr=IOT__USER_INFO:SHARED__IOT_USER_INFO_BETA ||\n'  # noqa
            '|| **YellowSkin ПП** | %%yellowskin://?url=https%3A%2F%2Fhamster.yandex.ru%2Fquasar%3Fsrcrwr%3DIOT_HOST%3Aiot-beta.quasar.yandex.ru%26srcrwr%3DIOT__USER_INFO%3ASHARED__IOT_USER_INFO_BETA%% ||\n'  # noqa
            '|| **Yellowskin ПП QR-код** | 0x0:https://disk.yandex.net/qr/?clean=1&text=yellowskin%3A%2F%2F%3Furl%3Dhttps%253A%252F%252Fhamster.yandex.ru%252Fquasar%253Fsrcrwr%253DIOT_HOST%25253Aiot-beta.quasar.yandex.ru%2526srcrwr%253DIOT__USER_INFO%253ASHARED__IOT_USER_INFO_BETA ||\n'  # noqa
            '|| **Конфиг для колонок** | %%{"uniProxyUrl": "wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-hamster/uni.ws?srcrwr=IOT_USER_INFO:VOICE__IOT_USER_INFO_BETA", "vinsUrl": "http://vins.hamster.alice.yandex.net/speechkit/app/pa/?srcrwr=IoT:iot-beta.quasar.yandex.net:80&srcrwr=SCENARIO_IOT_APPLY:iot-beta.quasar.yandex.net:80&srcrwr=IoTScenarios:iot-beta.quasar.yandex.net:80&srcrwr=SCENARIO_IOTSCENARIOS_APPLY:iot-beta.quasar.yandex.net:80&srcrwr=IoTVoiceDiscovery:iot-beta.quasar.yandex.net:80&srcrwr=SCENARIO_IOTVOICEDISCOVERY_APPLY:iot-beta.quasar.yandex.net:80"}%%  ||\n'  # noqa
            '|#\n'
        )

        runs = (
            '====Список ранов====\n'
            '<['
            '{runs}'
            ']>'
        ).format(runs="\n".join(map(StartrekClient.generate_run_line, created_runs)))

        instruction = (
            '====Инструкция по тестированию====\n'
            'https://wiki.yandex-team.ru/alicetesting/assessors-and-alice/'
        )
        return (
            '{header}\n\n'
            '{release_stand_configuration}\n\n'
            '{runs}\n\n'
            '{instruction}\n\n'
        ).format(
            header=header,
            release_stand_configuration=release_stand_configuration,
            runs=runs,
            instruction=instruction
        )
