import logging
from sandbox import sdk2
from sandbox.common import errors
from sandbox.sdk2.service_resources import SandboxTasksBinary


class IotDeviceGenerator(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        ydb_prefix = sdk2.parameters.String("ydb_prefix", default="/ru-prestable/quasar/beta/iotdb", required=True)
        ydb_endpoint = sdk2.parameters.String("ydb_endpoint", default="ydb-ru-prestable.yandex.net:2135", required=True)
        user_id = sdk2.parameters.String("user_id", default="", required=True)
        login = sdk2.parameters.String("login", default="", required=True)
        device_config = sdk2.parameters.String("device_config", default="", multiline=True, required=True)
        with sdk2.parameters.RadioGroup("Device Generator binary type") as release_type:
            release_type.values.stable = release_type.Value("stable", default=True)
            release_type.values.test = release_type.Value("test")
        ydb_token = sdk2.parameters.Vault("YDB token from Vault", description='"name" or "owner:name"', required=True)

    def on_save(self):
        attrs = {
            "target": "sandbox/projects/iot/IotDeviceGenerator",
            "release": self.Parameters.release_type or "stable"
        }
        res = SandboxTasksBinary.find(attrs=attrs).first()
        if res is not None:
            self.Requirements.tasks_resource = res.id
        else:
            raise errors.ResourceNotFound("Can't find binary for {} task ({} with attrs: {})".format(
                self.type.name, SandboxTasksBinary, attrs
            ))

    def on_execute(self):
        logging.info("The device generator has started")
        parameter_to_value = {
            "ydb_prefix" : self.Parameters.ydb_prefix,
            "ydb_endpoint" : self.Parameters.ydb_endpoint,
            "user_id" : self.Parameters.user_id,
            "login" : self.Parameters.login,
            "device_config" : self.Parameters.device_config,
        }

        resource = sdk2.Resource.find(
            type="IOT_DEVICE_GENERATOR_BINARY"
        ).first()
        if not resource:
            raise errors.TaskFailure("Could not find binary")
        path = str(sdk2.ResourceData(resource).path)
        logging.info("The path to executable is {}".format(path))
        with sdk2.helpers.ProcessLog(self, logger="shell_out") as pl:
            command_line_flags = [path]
            env = {"YDB_TOKEN": self.Parameters.ydb_token.data()}
            for name, value in parameter_to_value.items():
                command_line_flags.append("-{}={}".format(name, str(value)))
            logging.info("Command line arguments look like this: {}".format(command_line_flags))
            sdk2.helpers.subprocess.check_call(
                command_line_flags,
                stdout=pl.stdout, stderr=pl.stderr,
                env=env,
            )


class IotDeviceGeneratorBinary(sdk2.Resource):
    """"""
    arcadia_build_path = "alice/iot/bulbasaur/cmd/device_generator"
