# -*- coding: UTF-8 -*-
import json
import logging
import os.path

import sandbox.common.errors as ce
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.projects.iot.YdbDropTables import YdbDropTables
from sandbox.projects.iot.resources import IotYcBinary
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog
from sandbox.sdk2.service_resources import SandboxTasksBinary


class YcYdbRestore(sdk2.Task):
    """Restore ydb backup using yc ydb database restore"""

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 30000

        ydb_token = sdk2.parameters.Vault(
            "YDB token from Vault",
            description='"name" or "owner:name"',
            required=True,
        )
        yc_token = sdk2.parameters.Vault(
            "YC token from Vault",
            description='"name" or "owner:name"',
            required=True,
        )
        yc_cloud_id = sdk2.parameters.String(
            "Yandex cloud's cloud id",
            description="Cloud ID of the database: https://docs.yandex-team.ru/cloud/cli/concepts/core-properties.",
            required=True,
        )
        yc_folder_id = sdk2.parameters.String(
            "Yandex cloud's folder id",
            description="Folder ID of the database: https://docs.yandex-team.ru/cloud/cli/concepts/core-properties.",
            required=True,
        )

        with sdk2.parameters.Group("Source database parameters"):
            source_endpoint = sdk2.parameters.String(
                "Source database YDB endpoint",
                required=True,
            )
            source_database_name = sdk2.parameters.String(
                "Source database YDB name",
                description="The task will search for backups of this database.",
                required=True,
            )
            source_database_id = sdk2.parameters.String(
                "Source database YC ID",
                description="ID of the source database in yandex cloud. "
                            "Necessary, because ydb name and yc name of the same database can be different.",
                required=True,
            )

        with sdk2.parameters.Group("Target database parameters"):
            target_endpoint = sdk2.parameters.String(
                "Target database YDB endpoint",
                required=True,
            )
            target_database_name = sdk2.parameters.String(
                "Target database YDB name",
                description="The task will restore the backup to this database.",
                required=True,
            )
            target_database_id = sdk2.parameters.String(
                "Target database YC ID",
                description="ID of the target database in yandex cloud. "
                            "Necessary, because ydb name and yc name of the same database can be different.",
                required=True,
            )
            target_path = sdk2.parameters.String(
                "Target path",
                description="Path from database's root directiory to a folder where backup should be restored. "
                            "Empty by default.",
                default_value="",
                required=False,
            )
            drop_tables = sdk2.parameters.Bool(
                "Drop existing tables",
                default_value=True,
                required=True,
            )
            with drop_tables.value[True]:
                tables_to_drop = sdk2.parameters.List(
                    "List of tables to be dropped.",
                    default=[],
                    description="Empty means all tables will be dropped. Unknown paths are ignored.",
                )

    def on_save(self):
        attrs = {
            "target": "sandbox/projects/iot/YcYdbRestore",
            "release": "stable"
        }
        res = SandboxTasksBinary.find(attrs=attrs).first()
        if res is not None:
            self.Requirements.tasks_resource = res.id
        else:
            raise ce.ResourceNotFound("Can't find binary for %(type)s task (%(res)s with attrs: %(attrs)s)" % {
                "type": self.type.name,
                "res": SandboxTasksBinary.name,
                "attrs": attrs
            })

    def on_execute(self):
        if self.Parameters.drop_tables:
            with self.memoize_stage.drop_tables:
                params = {
                    "ydb_token": self.Parameters.ydb_token,
                    "endpoint": self.Parameters.target_endpoint,
                    "database": self.Parameters.target_database_name,
                    "path": self.Parameters.target_path,
                    "tables_to_drop": self.Parameters.tables_to_drop,
                }

                drop_tables_task = YdbDropTables(self, **params)
                drop_tables_task.save().enqueue()
                self.Context.drop_tables_task_id = drop_tables_task.id
                raise sdk2.WaitTask(
                    tasks=drop_tables_task,
                    statuses=list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                    wait_all=True,
                )

        yc_token = self.Parameters.yc_token.data()

        # Get and configure yc cli
        yc_cli_path = self.get_yc_cli_path()
        self.configure_yc_cli(
            path_to_cli=yc_cli_path,
            yc_token=yc_token,
            cloud_id=self.Parameters.yc_cloud_id,
            folder_id=self.Parameters.yc_folder_id,
        )
        logging.info('YC cli configured with cloud-id "%s" and folder-id "%s"',
                     self.Parameters.yc_cloud_id,
                     self.Parameters.yc_folder_id)

        # get latest backup
        latest_backup_id = self.get_latest_backup_id(yc_cli_path, self.Parameters.source_database_id)

        # restore the backup to the target db
        target_path = self.Parameters.target_database_name
        if self.Parameters.target_path:
            target_path = os.path.join(self.Parameters.target_database_name, self.Parameters.target_path.lstrip("/"))

        logging.info('Restoring the backup "%s" of "%s" to "%s"',
                     latest_backup_id,
                     self.Parameters.source_database_name,
                     target_path
                     )
        with ProcessLog(self, logger="common.log") as pl:
            args = [
                yc_cli_path,
                "ydb",
                "database",
                "restore",
                "--id=%s" % self.Parameters.target_database_id,
                "--backup-id=%s" % latest_backup_id,
            ]
            if self.Parameters.target_path:
                args += ["--target-path=%s" % self.Parameters.target_path.lstrip("/")],
            sp.check_call(args, shell=False, stdout=pl.stdout, stderr=pl.stdout)

    def get_latest_backup_id(self, path_to_cli, database_id):
        backups_json = sp.check_output([path_to_cli, "--format", "json", "ydb", "backup", "list"])
        # backups_json's structure:
        # [
        #     {
        #         "id": "...",
        #         "folder_id": "...",
        #         "database_id": "...",
        #         "created_at": "2022-01-18T14:06:20Z",
        #         "status": "READY",
        #         "backup_settings": {
        #             "name": "backup-name",
        #             "backup_time_to_live": "86400s"
        #         },
        #         "type": "USER",
        #         "size": "51205510996"
        #     },
        #     ...
        # ]
        backups = json.loads(backups_json)
        filtered_by_database = list(filter(lambda backup: backup["database_id"] == database_id, backups))
        latest_backup = max(filtered_by_database, key=lambda backup: backup["created_at"])
        return latest_backup["id"]

    def configure_yc_cli(self, path_to_cli, yc_token, cloud_id, folder_id):
        config_parameters = {
            "token": yc_token,
            "cloud-id": cloud_id,
            "folder-id": folder_id,
        }
        for k, v in config_parameters.items():
            sp.check_call([path_to_cli, "config", "set", k, v], shell=False)

    def get_yc_cli_path(self):
        logging.info("searching for yc cli binary")
        yc_cli_resource = IotYcBinary.find().first()
        if yc_cli_resource is None:
            raise ce.ResourceNotFound(ce.TaskError("Cannot find %s resource" % IotYcBinary.name))

        return str(sdk2.ResourceData(yc_cli_resource).path)
