# -*- coding: UTF-8 -*-

from sandbox import sdk2
import sandbox.common.errors as ce
from sandbox.projects.kikimr.resources import YdbCliBinary, YdbBackupData
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog


class YdbBackup2(sdk2.Task):
    """Backup YDB database using ya ydb tools dump"""

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 30000

        ydb_token = sdk2.parameters.Vault(
            "YDB token from Vault",
            description='"name" or "owner:name"',
            required=True,
        )

        with sdk2.parameters.Group("Backup parameters") as config_group:
            ydb_endpoint = sdk2.parameters.String(
                "YDB endpoint",
                description="host:port",
                default_value="ydb-ru-prestable.yandex.net:2135",
                required=True,
            )
            ydb_database = sdk2.parameters.String(
                "YDB database name",
                required=True,
            )
            path_in_database = sdk2.parameters.String(
                "Database path to a directory or a table to be dumped",
                default_value=".",
                description="empty means full dump",
            )
            schema_only = sdk2.parameters.Bool(
                "Only schema",
                default_value=False,
            )

        with sdk2.parameters.Group("Backup store parameters") as resource_group:
            store_forever = sdk2.parameters.Bool(
                "Store forever",
                default_value=False
            )
            with store_forever.value[False]:
                backup_ttl = sdk2.parameters.Integer(
                    "TTL",
                    description="days",
                    default_value=30,
                    required=True,
                )

    def on_execute(self):
        ydb_cli_path = self.get_ydb_cli_path()
        ydb_token = self.Parameters.ydb_token.data()

        resource = YdbBackupData(
            self,
            "YDB backup {}".format(self.Parameters.ydb_database),
            "backup",
            ydb_endpoint=self.Parameters.ydb_endpoint,
            ydb_database=self.Parameters.ydb_database,
            ydb_path=self.Parameters.path_in_database,
            ydb_schema_only=self.Parameters.schema_only,
            ttl=self.Parameters.backup_ttl,
        )
        if self.Parameters.store_forever:
            resource.ttl = "inf"

        data = sdk2.ResourceData(resource)
        data.path.mkdir(0o755, parents=True, exist_ok=True)

        args = [ydb_cli_path]
        args += ["--endpoint=%s" % self.Parameters.ydb_endpoint]
        args += ["--database=%s" % self.Parameters.ydb_database]
        args += ["tools", "dump"]
        if self.Parameters.path_in_database != "" and self.Parameters.path_in_database is not None:
            args += ["--path=%s" % self.Parameters.path_in_database]
        if self.Parameters.schema_only:
            args += ["--schema-only"]
        args += ["--output=%s" % str(data.path)]

        with ProcessLog(self, logger="ydb_backup") as pl:
            sp.check_call(args, env={"YDB_TOKEN": ydb_token}, shell=False, stdout=pl.stdout, stderr=pl.stderr)

        data.ready()

    def get_ydb_cli_path(self):
        ydb_cli_tar_resource = YdbCliBinary.find(
            attrs=dict(released="stable", platform="linux")
        ).first()
        if ydb_cli_tar_resource is None:
            raise ce.TaskError("Cannot find %s resource" % YdbCliBinary.name)

        ydb_cli_tar_path = str(sdk2.ResourceData(ydb_cli_tar_resource).path)

        with ProcessLog(self, logger="extract_ydb_cli") as pl:
            sp.check_call(["tar", "-xzf", ydb_cli_tar_path], shell=False, stdout=pl.stdout, stderr=pl.stderr)
        return str(sdk2.path.Path.cwd() / "ydb")
