# -*- coding: UTF-8 -*-

import sandbox.common.errors as ce
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.projects.iot.YdbBackup2 import YdbBackup2
from sandbox.projects.iot.YdbRestore2 import YdbRestore2
from sandbox.projects.kikimr.resources import YdbBackupData


class YdbBackupRestore(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.SSD

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group("Step 1. Backup from YDB") as backup_group:
            ydb_token = sdk2.parameters.Vault(
                "YDB token from Vault",
                description='"name" or "owner:name"',
                required=True,
            )

            ydb_endpoint = sdk2.parameters.String(
                "YDB endpoint",
                description="host:port",
                default_value="ydb-ru-prestable.yandex.net:2135",
                required=True,
            )

            ydb_database = sdk2.parameters.String(
                "YDB database name",
                required=True,
            )

            path_in_database = sdk2.parameters.String(
                "Path to a table or a directory to be backed up",
                description="empty means full database",
            )

            store_forever = sdk2.parameters.Bool(
                "Store forever",
                default_value=False
            )
            with store_forever.value[False]:
                backup_ttl = sdk2.parameters.Integer(
                    "TTL",
                    description="days",
                    default_value=180,
                    required=True,
                )

        with sdk2.parameters.Group("Step 2. Restore to YDB") as ydb_restore_group:
            ydb_restore_required = sdk2.parameters.Bool(
                "YDB restore required",
                default_value=False,
                required=True,
            )
            with ydb_restore_required.value[True]:
                ydb_restore_endpoint = sdk2.parameters.String(
                    "YDB restore endpoint",
                    description="host:port",
                    required=True,
                )

                ydb_restore_database = sdk2.parameters.String(
                    "YDB restore database name",
                    required=True,
                )

                ydb_restore_path_in_database = sdk2.parameters.String(
                    "Database directory to put restored data in",
                    description="Empty means root of database. Path will be created if necessary",
                )

                tables_to_ignore_in_ydb_restore = sdk2.parameters.List(
                    "List of tables to ignore during ydb restore phase", default=[],
                )

                ydb_drop_tables = sdk2.parameters.Bool(
                    "Drop tables",
                    description="Drop existing tables before restore",
                    default_value=False,
                )

                ydb_restore_indexes = sdk2.parameters.Bool(
                    "Restore indexes",
                    description="use with caution, task will fail if indexes already exist",
                    default_value=False,
                )

    def on_execute(self):
        with self.memoize_stage.backup:
            params = {
                "ydb_token": self.Parameters.ydb_token,
                "ydb_endpoint": self.Parameters.ydb_endpoint,
                "ydb_database": self.Parameters.ydb_database,
                "path_in_database": self.Parameters.path_in_database,
                "schema_only": False,
                "store_forever": self.Parameters.store_forever,
            }
            if not self.Parameters.store_forever:
                params["backup_ttl"] = self.Parameters.backup_ttl

            backup_task = YdbBackup2(self, **params)
            backup_task.save().enqueue()
            self.Context.backup_task_id = backup_task.id
            raise sdk2.WaitTask(
                backup_task,
                list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                wait_all=True
            )

        backup_task = self.find(None, id=self.Context.backup_task_id).first()
        if backup_task.status in ctt.Status.Group.BREAK + ctt.Status.Group.SCHEDULER_FAILURE:
            raise ce.TaskError("Subtask unsuccessfully finished with status %s" % backup_task.status)

        backup_resource = YdbBackupData.find(task=backup_task).first()
        if backup_resource is None:
            raise ce.TaskError("Subtask backup resource not found")

        # ydb restore may also be required
        if self.Parameters.ydb_restore_required:
            with self.memoize_stage.ydb_restore:
                ydb_restore_task = YdbRestore2(
                    self,
                    ydb_token=self.Parameters.ydb_token,
                    backup_resource=backup_resource,
                    endpoint=self.Parameters.ydb_restore_endpoint,
                    database=self.Parameters.ydb_restore_database,
                    path_in_database=self.Parameters.ydb_restore_path_in_database,
                    tables_to_ignore=self.Parameters.tables_to_ignore_in_ydb_restore,
                    restore_indexes=self.Parameters.ydb_restore_indexes,
                    drop_tables=self.Parameters.ydb_drop_tables,
                    check_only=False,
                )
                ydb_restore_task.save().enqueue()
                self.Context.ydb_restore_task_id = ydb_restore_task.id
                # we will wait for all tasks in the end

        # wait for ydb restore, if needed
        if self.Parameters.ydb_restore_required:
            ydb_restore_task = self.find(None, id=self.Context.ydb_restore_task_id).first()
            with self.memoize_stage.ydb_restore_wait:
                raise sdk2.WaitTask(
                    ydb_restore_task,
                    list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                    wait_all=True
                )
            if ydb_restore_task.status in ctt.Status.Group.BREAK + ctt.Status.Group.SCHEDULER_FAILURE:
                raise ce.TaskError("Subtask unsuccessfully finished with status %s" % ydb_restore_task.status)
