# -*- coding: UTF-8 -*-

from dateutil import tz

from sandbox import sdk2
import sandbox.common.errors as ce
import sandbox.common.types.task as ctt
from sandbox.projects.iot.YdbBackup2 import YdbBackup2
from sandbox.projects.iot.YdbRestore2 import YdbRestore2
from sandbox.projects.iot.YdbExportToYt import YdbExportToYt
from sandbox.projects.kikimr.resources import YdbBackupData


class YdbBackupRestore2(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):

        ydb_token = sdk2.parameters.Vault(
            "YDB token from Vault",
            description='"name" or "owner:name"',
            required=True,
        )
        yt_token = sdk2.parameters.Vault(
            "YT token from Vault",
            description='"name" or "owner:name"',
            required=True,
        )

        with sdk2.parameters.Group("Step 1. Backup from YDB") as backup_group:
            use_existing_backup = sdk2.parameters.Bool(
                "Use existing backup resource",
                default_value=False,
                required=True,
            )
            with use_existing_backup.value[True]:
                backup_resource = sdk2.parameters.Resource(
                    "Select resource with backup",
                    resource_type=YdbBackupData,
                    multiple=False,
                    required=True,
                )
            with use_existing_backup.value[False]:
                ydb_endpoint = sdk2.parameters.String(
                    "YDB endpoint",
                    description="host:port",
                    default_value="ydb-ru-prestable.yandex.net:2135",
                    required=True,
                )
                ydb_database = sdk2.parameters.String(
                    "YDB database name",
                    required=True,
                )
                store_forever = sdk2.parameters.Bool(
                    "Store forever",
                    default_value=False
                )
                with store_forever.value[False]:
                    backup_ttl = sdk2.parameters.Integer(
                        "TTL",
                        description="days",
                        default_value=180,
                        required=True,
                    )

        with sdk2.parameters.Group("Step 2. Restore to YDB") as restore_group:
            ydb_restore_endpoint = sdk2.parameters.String(
                "YDB restore endpoint",
                description="host:port",
                required=True,
            )
            ydb_restore_database = sdk2.parameters.String(
                "YDB restore database name",
                required=True,
            )
            ydb_restore_path_in_database = sdk2.parameters.String(
                "Database directory to put restored data in",
                description="Empty means root of database. Path will be created if necessary",
            )
            tables_to_ignore_in_ydb_restore = sdk2.parameters.List(
                "List of tables to ignore during ydb restore phase", default=[],
            )
            ydb_restore_indexes = sdk2.parameters.Bool(
                "Restore indexes",
                description="use with caution, task will fail if indexes already exist",
                default_value=False,
            )

        with sdk2.parameters.Group("Step 3. Export to YT") as export_group:
            proxy = sdk2.parameters.String(
                "YT proxy (cluster)",
                default_value="hahn",
                required=True,
            )
            destination = sdk2.parameters.String(
                "Destination folder",
                description="available placeholders: "
                            "%export_date% = export date in format YYYY-MM-DD; "
                            "%now% = current timestamp; ",
                required=True,
            )
            exclude_pattern = sdk2.parameters.String(
                "Exclude pattern",
                description="Pattern (PCRE) for paths excluded from export operation",
                default_value="",
            )
            use_type_v3 = sdk2.parameters.Bool(
                "Use YT type_v3",
                default=True,
                required=True,
            )

            additional_export_required = sdk2.parameters.Bool(
                "Additional export required",
                default_value=False,
                required=True,
            )
            with additional_export_required.value[True]:
                additional_export_proxy = sdk2.parameters.String(
                    "Additional export YT proxy (cluster)",
                    default_value="arnold",
                    required=True,
                )
                additional_export_destination = sdk2.parameters.String(
                    "Additional export destination folder",
                    description="available placeholders: "
                                "%export_date% = export date in format YYYY-MM-DD; "
                                "%now% = current timestamp; ",
                    required=True,
                )

    def on_execute(self):
        # Step 1 - Acquiring backup resource
        if not self.Parameters.use_existing_backup:
            with self.memoize_stage.backup:
                params = {
                    "ydb_token": self.Parameters.ydb_token,
                    "ydb_endpoint": self.Parameters.ydb_endpoint,
                    "ydb_database": self.Parameters.ydb_database,
                    "path_in_database": self.Parameters.path_in_database,
                    "schema_only": False,
                    "store_forever": self.Parameters.store_forever,
                }
                if not self.Parameters.store_forever:
                    params["backup_ttl"] = self.Parameters.backup_ttl

                backup_task = YdbBackup2(self, **params)
                backup_task.save().enqueue()
                self.Context.backup_task_id = backup_task.id
                raise sdk2.WaitTask(
                    backup_task,
                    list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                    wait_all=True
                )

        backup_resource = self.Parameters.backup_resource
        if not self.Parameters.use_existing_backup:
            backup_task = self.find(None, id=self.Context.backup_task_id).first()
            if backup_task.status in ctt.Status.Group.BREAK + ctt.Status.Group.SCHEDULER_FAILURE:
                raise ce.TaskError("can't get backup: backup task finished with status %s" % backup_task.status)
            backup_resource = YdbBackupData.find(task=backup_task).first()
        if backup_resource is None:
            raise ce.TaskError("Backup resource is not found")

        # Step 2 - Restoring to YDB
        with self.memoize_stage.ydb_restore:
            ydb_restore_task = YdbRestore2(
                self,
                ydb_token=self.Parameters.ydb_token,
                backup_resource=backup_resource,
                endpoint=self.Parameters.ydb_restore_endpoint,
                database=self.Parameters.ydb_restore_database,
                path_in_database=self.Parameters.ydb_restore_path_in_database,
                tables_to_ignore=self.Parameters.tables_to_ignore_in_ydb_restore,
                restore_indexes=self.Parameters.ydb_restore_indexes,
                drop_tables=True,
                check_only=False,
            )
            ydb_restore_task.save().enqueue()
            self.Context.ydb_restore_task_id = ydb_restore_task.id
            # restoring is crucial
            raise sdk2.WaitTask(
                ydb_restore_task,
                list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                wait_all=True
            )
        restore_task = self.find(None, id=self.Context.ydb_restore_task_id).first()
        if restore_task.status in ctt.Status.Group.BREAK + ctt.Status.Group.SCHEDULER_FAILURE:
            raise ce.TaskError("ydb restore task finished with status %s" % restore_task.status)

        with self.memoize_stage.yt_export:
            export_date = backup_resource.created.astimezone(tz.tzutc()).replace(tzinfo=None).strftime("%Y-%m-%d")
            export_task = YdbExportToYt(
                self,
                release_type="stable",
                export_date=export_date,

                ydb_token=self.Parameters.ydb_token,
                ydb_endpoint=self.Parameters.ydb_restore_endpoint,
                ydb_database=self.Parameters.ydb_restore_database,

                yt_token=self.Parameters.yt_token,
                proxy=self.Parameters.proxy,
                destination=self.Parameters.destination,
                exclude_pattern=self.Parameters.exclude_pattern,
                use_type_v3=self.Parameters.use_type_v3,
            )
            export_task.save().enqueue()
            self.Context.yt_export_task_id = export_task.id
            # we will wait for task completion separately

        if self.Parameters.additional_export_required:
            with self.memoize_stage.additional_yt_export:
                export_date = backup_resource.created.astimezone(tz.tzutc()).replace(tzinfo=None).strftime("%Y-%m-%d")
                additional_export_task = YdbExportToYt(
                    self,
                    release_type="stable",
                    export_date=export_date,

                    ydb_token=self.Parameters.ydb_token,
                    ydb_endpoint=self.Parameters.ydb_restore_endpoint,
                    ydb_database=self.Parameters.ydb_restore_database,

                    yt_token=self.Parameters.yt_token,
                    proxy=self.Parameters.additional_export_proxy,
                    destination=self.Parameters.additional_export_destination,
                    exclude_pattern=self.Parameters.exclude_pattern,
                    use_type_v3=self.Parameters.use_type_v3,
                )
                additional_export_task.save().enqueue()
                self.Context.additional_yt_export_task_id = additional_export_task.id
                # we will wait for task completion separately

        with self.memoize_stage.yt_export_wait:
            yt_export_task = self.find(None, id=self.Context.yt_export_task_id).first()
            if yt_export_task.status in ctt.Status.Group.BREAK + ctt.Status.Group.SCHEDULER_FAILURE:
                raise ce.TaskError("ydb restore task finished with status %s" % yt_export_task.status)

        if self.Parameters.additional_export_required:
            with self.memoize_stage.additional_yt_export_wait:
                additional_yt_export_task = self.find(None, id=self.Context.additional_yt_export_task_id).first()
                if additional_yt_export_task.status in ctt.Status.Group.BREAK + ctt.Status.Group.SCHEDULER_FAILURE:
                    raise ce.TaskError("ydb restore task finished with status %s" % additional_yt_export_task.status)
