# -*- coding: UTF-8 -*-

import sandbox.common.errors as ce
from datetime import datetime
import json
import time
import logging
import os

from sandbox import sdk2
from sandbox.sdk2.service_resources import SandboxTasksBinary
from sandbox.projects.kikimr.resources import YdbCliBinary
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog


class YdbExportToYt(sdk2.Task):
    """Export existing YDB database to YT"""

    YDB_OPERATION_SUCCESS_STATUS = "PROGRESS_DONE"

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.RadioGroup("YdbExportToYt binary type") as release_type:
            release_type.values.stable = release_type.Value("stable", default=True)
            release_type.values.test = release_type.Value("test")

        export_date = sdk2.parameters.String(
            "Export date",
            description="YYYY-MM-DD format. If empty, current date will be used.",
            default_value="",
        )

        with sdk2.parameters.Group("YDB parameters") as ydb_config_group:
            ydb_token = sdk2.parameters.Vault(
                "YDB token from Vault",
                description='"name" or "owner:name"',
                required=True,
            )
            ydb_endpoint = sdk2.parameters.String(
                "YDB endpoint",
                description="host:port",
                default_value="ydb-ru-prestable.yandex.net:2135",
                required=True,
            )
            ydb_database = sdk2.parameters.String(
                "YDB database name",
                required=True,
            )

        with sdk2.parameters.Group("Export parameters") as export_config_group:
            yt_token = sdk2.parameters.Vault(
                "YT token from Vault",
                description='"name" or "owner:name"',
                required=True,
            )
            yt_proxy = sdk2.parameters.String(
                "YT proxy (cluster)",
                default="hahn",
                required=True,
            )
            destination = sdk2.parameters.String(
                "Destination folder",
                description="available placeholders: "
                            "%export_date% = export date in format YYYY-MM-DD; "
                            "%now% = current timestamp; ",
                required=True,
            )
            drop_yt_tables = sdk2.parameters.Bool(
                "Drop YT tables",
                description="Drop existing tables from YT before export",
                default=False,
            )
            exclude_pattern = sdk2.parameters.String(
                "Exclude pattern",
                description="Pattern (PCRE) for paths excluded from export operation",
                default_value="",
            )
            use_type_v3 = sdk2.parameters.Bool(
                "Use YT type_v3",
                default=True,
                required=True,
            )
            retries = sdk2.parameters.Integer(
                "Count of retries",
                default=120,
                required=True,
            )
            delay = sdk2.parameters.Integer(
                "Delay between retries in seconds",
                default=30,
                required=True,
            )

        with sdk2.parameters.Group("Reactor parameters") as reactor_config_group:
            publish_reactor_artifact = sdk2.parameters.Bool(
                "Publish reactor artifact",
                default_value=False,
                required=True,
            )
            with publish_reactor_artifact.value[True]:
                reactor_token = sdk2.parameters.Vault(
                    "Reactor token from Vault",
                    description='"name" or "owner:name"',
                    required=True,
                )
                reactor_artifact_path = sdk2.parameters.String(
                    "Reactor artifact path",
                    default_value="",
                    required=True,
                )

        # Latest folder
        with sdk2.parameters.Group("Latest folder parameters") as duplicate_group:
            copy_to_latest = sdk2.parameters.Bool(
                'Duplicate result to "latest" folder in the same directory. '
                'All contents of the "latest" directory will be rewritten',
                default_value=False,
                required=True,
            )
            with copy_to_latest.value[True]:
                latest_publish_reactor_artifact = sdk2.parameters.Bool(
                    "Publish reactor artifact",
                    default_value=False,
                    required=True,
                )
                with latest_publish_reactor_artifact.value[True]:
                    latest_reactor_token = sdk2.parameters.Vault(
                        "Reactor token from Vault",
                        description='"name" or "owner:name"',
                        required=True,
                    )
                    latest_reactor_artifact_path = sdk2.parameters.String(
                        "Reactor artifact path",
                        default_value="",
                        required=True,
                    )

    def on_save(self):
        attrs = {
            "target": "sandbox/projects/iot/YdbExportToYt",
            "release": self.Parameters.release_type or "stable"
        }
        res = SandboxTasksBinary.find(attrs=attrs).first()
        if res is not None:
            self.Requirements.tasks_resource = res.id
        else:
            raise ce.ResourceNotFound("Can't find binary for %(type)s task (%(res)s with attrs: %(attrs)s)" % {
                "type": self.type.name,
                "res": SandboxTasksBinary.name,
                "attrs": attrs
            })

    def on_execute(self):
        import yt.wrapper as yt

        export_date = self.Parameters.export_date
        if export_date == "":
            export_date = datetime.utcnow().strftime("%Y-%m-%d")

        destination = self.Parameters.destination \
            .replace("%now%", datetime.utcnow().strftime("%Y%m%dT%H%M%S")) \
            .replace("%export_date%", export_date)

        ydb_cli_path = self.get_ydb_cli_path()
        args = [ydb_cli_path]
        args += ["--endpoint=%s" % self.Parameters.ydb_endpoint]
        args += ["--database=%s" % self.Parameters.ydb_database]
        args += ["export", "yt"]
        args += ["--item", "dst=%s,src=%s" % (destination, self.Parameters.ydb_database)]
        if self.Parameters.exclude_pattern:
            args += ['--exclude="%s"' % self.Parameters.exclude_pattern]
        if self.Parameters.description:
            args += ['--description="%s"' % self.Parameters.description]
        if self.Parameters.use_type_v3:
            args += ["--use-type-v3"]
        args += ["--format", "proto-json-base64"]

        ydb_token = self.Parameters.ydb_token.data()
        yt_token = self.Parameters.yt_token.data()
        yt_proxy = self.Parameters.yt_proxy
        env = {"YDB_TOKEN": ydb_token, "YT_TOKEN": yt_token, "YT_PROXY": yt_proxy}

        if self.Parameters.drop_yt_tables:
            self.remove_yt_directory(destination)

        # call export
        export_info = json.loads(sp.check_output(args, env=env))
        # forget operation
        self.forget_operation(ydb_cli_path, export_info["id"], env=env)
        # create reactor artifact
        if self.Parameters.publish_reactor_artifact:
            self.create_reactor_artifact(destination)

        # copy to "latest" folder
        if self.Parameters.copy_to_latest:
            path_to_latest = self.Parameters.destination \
                                 .replace("%now%", "") \
                                 .replace("%export_date%", "")
            path_to_latest = os.path.join(path_to_latest, "latest")
            yt.config.set_proxy(yt_proxy)
            yt.config["token"] = self.Parameters.yt_token.data()
            yt.copy(
                source_path=destination,
                destination_path=path_to_latest,
                recursive=True,
                force=True
            )

    def remove_yt_directory(self, path):
        import yt.wrapper as yt
        # remove tables from destination
        yt.config.set_proxy(self.Parameters.yt_proxy)
        yt.config["token"] = self.Parameters.yt_token.data()
        yt.remove(yt.YPath(path), force=True, recursive=True)
        logging.info('Dropped {}'.format(self.Parameters.destination))

    def get_ydb_cli_path(self):
        ydb_cli_tar_resource = YdbCliBinary.find(
            attrs=dict(released="stable", platform="linux")
        ).first()
        if ydb_cli_tar_resource is None:
            raise ce.TaskError("Cannot find %s resource" % YdbCliBinary.name)

        ydb_cli_tar_path = str(sdk2.ResourceData(ydb_cli_tar_resource).path)

        with ProcessLog(self, logger="extract_ydb_cli") as pl:
            sp.check_call(["tar", "-xzf", ydb_cli_tar_path], shell=False, stdout=pl.stdout, stderr=pl.stderr)
        return str(sdk2.path.Path.cwd() / "ydb")

    def get_operation_status(self, ydb_cli_path, operation_id, env):
        operation_str = json.loads(sp.check_output([
            ydb_cli_path,
            '--endpoint', self.Parameters.ydb_endpoint,
            '--database', self.Parameters.ydb_database,
            'operation', 'get', operation_id,
            '--format', "proto-json-base64"], env=env))
        operation_status = operation_str["metadata"]["progress"]
        logging.info('Operation', operation_id, 'status is', operation_status)
        return operation_status

    def forget_operation(self, ydb_cli_path, operation_id, env):
        checks_number = self.Parameters.retries
        delay = self.Parameters.delay
        success = False
        while checks_number > 0 and not success:
            logging.info('Trying to forget operation, checks left: ', checks_number)
            operation_status = self.get_operation_status(ydb_cli_path, operation_id, env=env)
            if operation_status == self.YDB_OPERATION_SUCCESS_STATUS:
                sp.check_call([
                    ydb_cli_path,
                    '--endpoint', self.Parameters.ydb_endpoint,
                    '--database', self.Parameters.ydb_database,
                    'operation', 'forget', operation_id], env=env)
                success = True
                logging.info('Operation forgotten: ', checks_number)
                break

            checks_number -= 1
            time.sleep(delay)

        if not success:
            raise Exception("Operation was not completed successfully")

    def create_reactor_artifact(self, yt_path):
        from reactor_client.helper.artifact_instance import ArtifactInstanceBuilder
        from reactor_client.reactor_api import ReactorAPIClientV1

        client = ReactorAPIClientV1(base_url="https://reactor.yandex-team.ru",
                                    token=self.Parameters.reactor_token.data())

        artifact_instance = ArtifactInstanceBuilder() \
            .set_artifact_path(self.Parameters.reactor_artifact_path) \
            .set_yt_path(self.Parameters.yt_proxy, yt_path) \
            .set_attributes({}) \
            .set_user_time(datetime.now())

        client.artifact_instance.instantiate(
            artifact_identifier=artifact_instance.artifact_identifier,
            metadata=artifact_instance.value,
            attributes=artifact_instance.attributes,
            user_time=artifact_instance.user_time,
            create_if_not_exist=True
        )
