# -*- coding: UTF-8 -*-

import logging

import sandbox.common.errors as ce
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.projects.iot.YdbDropTables import YdbDropTables
from sandbox.projects.kikimr.resources import YdbCliBinary, YdbBackupData
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog


class YdbRestore2(sdk2.Task):
    """Check and restore previously saved YDB backup using ya ydb tools restore"""

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 30000

        ydb_token = sdk2.parameters.Vault(
            "YDB token from Vault",
            description='"name" or "owner:name"',
            required=True,
        )

        with sdk2.parameters.Group("Choose resource with backup"):
            backup_resource = sdk2.parameters.Resource(
                "Select resource with backup",
                resource_type=YdbBackupData,
                multiple=False,
                required=True,
            )

        with sdk2.parameters.Group("Restore parameters") as config_group:
            endpoint = sdk2.parameters.String(
                "YDB endpoint",
                description="host:port",
                default_value="ydb-ru-prestable.yandex.net:2135",
                required=True,
            )
            database = sdk2.parameters.String(
                "YDB database name",
                required=True,
            )
            path_in_database = sdk2.parameters.String(
                "Database path to a destination directory where restored directory or table will be placed",
                default_value="./",
                description="Empty means root of database. Path will be created if necessary.",
            )
            path_in_backup = sdk2.parameters.String(
                "Path in backup data",
                description="Empty means full backup data",
            )
            tables_to_ignore = sdk2.parameters.List(
                "List of tables to ignore during restore", default=[],
            )
            drop_tables = sdk2.parameters.Bool(
                "Drop existing tables",
                default_value=True,
                required=True,
            )
            with drop_tables.value[True]:
                tables_to_drop = sdk2.parameters.List(
                    "List of tables to be dropped.",
                    default=[],
                    description="Empty means all tables will be dropped. Unknown paths are ignored.",
                )
            check_only = sdk2.parameters.Bool(
                "Only check",
                description="Do not restore tables, only check that:\n"
                            "\t- all dumped tables exist in database\n"
                            "\t- all dumped table schemes are the same as in database"
                            "Ignored if drop_tables parameter is true",
                default_value=False,
            )
            restore_indexes = sdk2.parameters.Bool(
                "Restore indexes",
                description="use with caution, task will fail if table indexes already exist",
                default_value=True,
            )

    def on_execute(self):
        if self.Parameters.drop_tables:
            with self.memoize_stage.drop_tables:
                params = {
                    "ydb_token": self.Parameters.ydb_token,
                    "endpoint": self.Parameters.endpoint,
                    "database": self.Parameters.database,
                    "tables_to_drop": self.Parameters.tables_to_drop,
                }

                drop_tables_task = YdbDropTables(self, **params)
                drop_tables_task.save().enqueue()
                self.Context.drop_tables_task_id = drop_tables_task.id
                raise sdk2.WaitTask(
                    drop_tables_task,
                    list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                    wait_all=True
                )

        ydb_cli_path = self.get_ydb_cli_path()
        ydb_token = self.Parameters.ydb_token.data()

        backup_data = sdk2.ResourceData(self.Parameters.backup_resource)

        args = [ydb_cli_path]
        args += ["--endpoint=%s" % self.Parameters.endpoint]
        args += ["--database=%s" % self.Parameters.database]
        args += ["tools", "restore"]
        if self.Parameters.path_in_database != "" and self.Parameters.path_in_database is not None:
            args += ["--path=%s" % self.Parameters.path_in_database]
        else:
            args += ["--path=./"]

        backup_path = backup_data.path
        if self.Parameters.path_in_backup != "" and self.Parameters.path_in_backup is not None:
            backup_path = backup_path.joinpath(self.Parameters.path_in_backup)
        backup_path.chmod(0o777)

        logging.info("Will restore tables from path %s" % backup_path)
        logging.info("Will ignore tables [%s]" % " ".join(self.Parameters.tables_to_ignore))

        for table_schema_path in list(backup_data.path.glob("**/scheme.pb")):
            table_directory_path = table_schema_path.parent
            table_path = table_directory_path.relative_to(backup_data.path)
            logging.info("  Found %s" % table_schema_path.relative_to(backup_data.path))

            if not table_schema_path.is_file():
                logging.warn("      It's not file, skip")
                continue

            if str(table_path) in self.Parameters.tables_to_ignore:
                logging.info("      It's in ignored tables, will remove it from backup data")
                self.rmdir(table_directory_path)

        args += ["--input=%s" % str(backup_path)]
        if self.Parameters.check_only and not self.Parameters.drop_tables:
            args += ["--dry-run"]
        if not self.Parameters.restore_indexes:
            args += ["--restore-indexes=0"]

        with ProcessLog(self, logger="ydb_backup") as pl:
            sp.check_call(args, env={"YDB_TOKEN": ydb_token}, shell=False, stdout=pl.stdout, stderr=pl.stdout)

    def get_ydb_cli_path(self):
        ydb_cli_tar_resource = YdbCliBinary.find(
            attrs=dict(released="stable", platform="linux")
        ).first()
        if ydb_cli_tar_resource is None:
            raise ce.TaskError("Cannot find %s resource" % YdbCliBinary.name)

        ydb_cli_tar_path = str(sdk2.ResourceData(ydb_cli_tar_resource).path)

        with ProcessLog(self, logger="extract_ydb_cli") as pl:
            sp.check_call(["tar", "-xzf", ydb_cli_tar_path], shell=False, stdout=pl.stdout, stderr=pl.stderr)
        return str(sdk2.path.Path.cwd() / "ydb")

    def rmdir(self, path):
        if not path.is_dir():
            return

        path.chmod(0o777)
        for child in list(path.iterdir()):
            child.chmod(0o666)
            if child.is_dir():
                self.rmdir(child)
            else:
                child.unlink()
        path.rmdir()
