import time
import hashlib
import logging
import requests
from urlparse import urljoin
import json

logger = logging.getLogger(__name__)


def timestamp_ms():
    return int(time.time() * 1000)


class TuyaException(Exception):
    pass


class TuyaTokenExpiredException(TuyaException):
    def __init__(self, message=None):
        self.message = 'Tuya token is expired' if message is None else message


def process_tuya_json(resp):
    if resp.status_code != 200:
        raise TuyaException('Tuya API returns HTTP error {}'.format(resp.status))

    try:
        result = json.loads(resp.text())
    except Exception as e:
        raise TuyaException('JSONDecodeError while parsing Tuya response: {}'.format(e))

    for field in ['success', 'result']:
        if result.get(field) is None:
            raise TuyaException('No [{}] field in Tuya response. Response: {}'.format(field, result))

    if result.get('success') is False:
        if result.get('code') == 1010:
            raise TuyaTokenExpiredException()
        raise TuyaException('Operation failed. Reason: [{}] {}'.format(result.get('code'), result.get('msg')))

    return result


def with_token_refresh(func):
    def payload(self, *a, **kw):
        self.refresh_token()
        try:
            return func(self, *a, **kw)
        except TuyaTokenExpiredException:
            logger.warning('Catch TuyaTokenExpireException while executing {}. Trying to get token and retry operation'.format(func.__name__))
            self.token = self._get_token()
            return func(self, *a, **kw)
    return payload


class TuyaToken(object):
    def __init__(self, value, expiration_ts, creation_ts, refresh_token):
        self.value = value
        self.expiration_ts = expiration_ts
        self.creation_ts = creation_ts
        self.refresh_token = refresh_token

    @property
    def is_expired(self):
        return self.expiration_ts <= timestamp_ms()


class TuyaClient(object):
    def __init__(self, client_id, secret, host):
        self.client_id = client_id
        self.secret = secret
        self.tuya_host = host
        self.token = None

    def _request_url(self, path, api_ver=1.0):
        return urljoin('https://{}/v{}/'.format(self.tuya_host, api_ver), path)

    @property
    def _get_headers(self):
        ts = timestamp_ms()
        return {'client_id': self.client_id,
                'sign': self._sign(ts, self.token.value),
                't': str(ts),
                'access_token': self.token.value}

    @property
    def _post_headers(self):
        return dict(self._get_headers, **{'Content-Type': 'application/json'})

    def _sign(self, ts=None, token=''):
        if ts is None:
            ts = timestamp_ms()
        return hashlib.md5(''.join([self.client_id, token, self.secret, str(ts)]).encode('utf-8')).hexdigest().upper()

    def _get_token(self):
        ts = timestamp_ms()
        headers = {'t': str(ts), 'sign': self._sign(ts=ts), 'client_id': self.client_id}

        resp = requests.get(url=self._request_url('token?grant_type=1'),
                            headers=headers,
                            verify_ssl=False)

        try:
            result = process_tuya_json(resp)
        except TuyaException as e:
            raise TuyaException('Failed to get token. Reason: {}'.format(e))

        logger.info("Token request result: {}".format(result))
        return TuyaToken(value=result['result']['access_token'],
                         expiration_ts=ts + int(result['result']['expire_time']) * 1000,
                         creation_ts=ts,
                         refresh_token=result['result']['refresh_token'])

    def refresh_token(self):
        if self.token is None or self.token.is_expired:
            self.token = self._get_token()

    @with_token_refresh
    def post_ir_action(self, hub_id, payload):
        logger.info('Executing action {} for IR AC {}'.format(payload, hub_id))

        resp = requests.post(url=self._request_url('infrareds/{}/send-ackeys'.format(hub_id)),
                             headers=self._post_headers,
                             json=payload,
                             verify_ssl=False)

        try:
            process_tuya_json(resp)
        except TuyaException as e:
            raise TuyaException('Executing action {} for IR {}. Reason: {}'.format(payload, hub_id, e))

    @with_token_refresh
    def get_ac_state(self, hub_id, control_id):
        logger.info('Trying to get state for IR AC {}'.format(hub_id))

        resp = requests.get(url=self._request_url('infrareds/{}/remotes/{}/ac/status'.format(hub_id, control_id)),
                            headers=self._get_headers,
                            verify_ssl=False)

        try:
            state = process_tuya_json(resp)
        except TuyaException as e:
            raise TuyaException('Getting state for control {} from IR hub {}. Reason: {}'.format(control_id, hub_id, e))

        result = state['result']

        return {
            'power': result.get('power', 99),
            'mode': result.get('mode', 99),
            'wind': result.get('wind', 99),
            'temp': result.get('temp', 99),
        }
