import logging
import os
import shutil

import sandbox.projects.common.arcadia.sdk as arcadiasdk
import sandbox.projects.common.constants.constants as sdk_constants
import sandbox.projects.common.vcs.aapi as aapi
import sandbox.sdk2 as sdk2


class Platform(object):
    Darwin = 'DEFAULT-DARWIN-X86_64'
    Linux = 'DEFAULT-LINUX-X86_64'


class IrtBuild(sdk2.Task):
    class Parameters(sdk2.Parameters):
        debug = sdk2.parameters.Bool('Debug build', required=False)

        with sdk2.parameters.RadioGroup('Platform') as platform:
            platform.values['Linux'] = platform.Value(
                value=Platform.Linux, default=True)
            platform.values['OS X'] = platform.Value(value=Platform.Darwin)

        arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None, required=False)
        arcanum_review = sdk2.parameters.String("Arcanum review", default=None, required=False)
        checkout_arcadia_from_url = sdk2.parameters.String("Svn url for arc with revision", default=sdk2.svn.Arcadia.ARCADIA_TRUNK_URL, required=True)
        arcadia_patch = sdk2.parameters.String('Apply patch (text diff or rbtorrent)', multiline=True, default='')
        skip_on_empty_arcadia_patch = sdk2.parameters.Bool(
            "Skip execution if not a patched run."
            " Useful for skipping run in a patch-less run of pre-commit check in testenv",
            default_value=False
        )

    def should_skip(self):
        if getattr(self.Parameters, 'skip_on_empty_arcadia_patch', False) and not getattr(self.Parameters, 'arcadia_patch', ''):
            self.set_info(
                'Task execution skipped: '
                '"skip_on_empty_arcadia_patch" parameter is True,'
                ' but "arcadia_patch" context field is missing or empty'
                ' (task must be run in a patched run)'
            )
            return True
        return False

    @classmethod
    def _ensure_list(cls, candidate):
        return candidate if isinstance(candidate, list) else [candidate]

    def copy_files(self, files, revision=None, arc_review=None):
        if arc_review:
            patch = 'arc:{}'.format(arc_review)
        elif getattr(self.Parameters, 'arcadia_patch', ''):
            patch = self.Parameters.arcadia_patch
        else:
            patch = None

        arcadia_url = getattr(self.Parameters, 'checkout_arcadia_from_url', sdk2.svn.Arcadia.ARCADIA_TRUNK_URL)
        if revision is not None:
            arcadia_url = sdk2.svn.Arcadia.replace(arcadia_url, revision=revision)

        with arcadiasdk.mount_arc_path(arcadia_url) as a_arcadia_path:
            if patch:
                arcadiasdk.apply_patch(self, a_arcadia_path, patch, self.path())

            for src, dst in files.items():
                shutil.copy(os.path.join(a_arcadia_path, src), dst)

    def build(self, targets, destination, platform=Platform.Linux, revision=None, arc_review=None, debug=False):
        """
        Building ya target. Will move to abstract class in future

        :param targets: Target to build
        :param destination:
        :param platform: Platform for build
        :param revision: Arcadia revision or None for HEAD
        :param arc_review: Review id in arcanum
        :param debug: Enable debug
        :return:
        """
        logger = logging.getLogger(__name__)

        revision = revision or aapi.ArcadiaApi.svn_head()
        logging.info('__{}__{}__'.format(revision, type(revision)))
        logger.info('Building next targets:\n{}\nUsing {} revision'.format('\n'.join('\t{}'.format(t) for t in targets), revision))

        if arc_review:
            logger.info('Apply review patch {}'.format(arc_review))
            patch = 'arc:{}'.format(arc_review)
        elif getattr(self.Parameters, 'arcadia_patch', ''):
            logger.info('Apply arcadia patch {}'.format(self.Parameters.arcadia_patch))
            patch = self.Parameters.arcadia_patch
        else:
            patch = None

        with arcadiasdk.mount_arc_path(sdk2.svn.Arcadia.replace(getattr(self.Parameters, 'checkout_arcadia_from_url', sdk2.svn.Arcadia.ARCADIA_TRUNK_URL), revision=revision)) as a_arcadia_path:
            if patch:
                arcadiasdk.apply_patch(self, a_arcadia_path, patch, self.path())

            arcadiasdk.do_build(
                build_system=sdk_constants.SEMI_DISTBUILD_BUILD_SYSTEM,
                source_root=a_arcadia_path,
                revision=revision,
                targets=self._ensure_list(targets),
                target_platform=platform,
                results_dir=destination,
                clear_build=False,
                patch=patch,
                build_type=sdk_constants.DEBUG_BUILD_TYPE if debug else sdk_constants.RELEASE_BUILD_TYPE,
                force_vcs_info_update=True,
                def_flags='',
            )

        return [
            os.path.join(destination, target, build)
            for target in targets
            for build in os.listdir(os.path.join(destination, target))
            if not os.path.isdir(os.path.join(destination, target, build))
        ], revision
