import os
import logging

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.resource as ctr
import sandbox.projects.common.binary_task as binary_task

from sandbox.projects.irt.bannerland.backend import BannerLandWebBackendBinary, BannerLandWebStatic


class BannerlandBinMakePocket(sdk2.Resource):
    dyn_preprod_checked = sdk2.Attributes.Bool("Checked on dyn preprod", required=True, default=False)
    perf_preprod_checked = sdk2.Attributes.Bool("Checked on perf preprod", required=True, default=False)
    recom_preprod_checked = sdk2.Attributes.Bool("Checked on recom preprod", required=True, default=False)


class BannerlandRunMakePocket(binary_task.LastBinaryTaskKosherReleaseMixin, sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 800

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        bannerland_task_type = sdk2.parameters.String('Task type', choices=[(s, s) for s in ['perf', 'dyn', 'recom']], default='perf')
        run_env = sdk2.parameters.String('Run environment', choices=[(s, s) for s in ['user', 'preprod', 'prod']], default='user')
        bannerland_make_banners_mode = sdk2.parameters.String('make_banners mode', choices=[(s, s) for s in ['yt', 'blrt']], default='yt')
        with run_env.value['user']:
            yt_root = sdk2.parameters.String('YT root')
            maker_id = sdk2.parameters.Integer('Pocket maker resource id')

        use_last_taskbin = sdk2.parameters.Bool('Use last task binary resource', default=True)
        use_native_sandbox_release = sdk2.parameters.Bool('Use on_save of LastBinaryTaskKosherReleaseMixin', default=False)
        with use_last_taskbin.value[True]:
            with sdk2.parameters.RadioGroup('Task binary release type') as taskbin_release_type:
                taskbin_release_type.values.stable = taskbin_release_type.Value('stable', default=True)
                taskbin_release_type.values.none = taskbin_release_type.Value('none')
        with use_last_taskbin.value[False]:
            taskbin_resource = sdk2.parameters.Resource('Task binary resource', default=None)
        yt_token = sdk2.parameters.YavSecret(
            'YAV secret identifier YT',
            default='sec-01cqpqffpbg39j62zmngfzcx2w#yt_plato'
        )
        yql_token = sdk2.parameters.YavSecret(
            'YAV secret identifier YQL',
            default='sec-01cqpqffpbg39j62zmngfzcx2w#yql_robot_bm_admin'
        )

    def on_save(self):
        if self.Parameters.use_native_sandbox_release:
            super(BannerlandRunMakePocket, self).on_save()
        elif self.Parameters.use_last_taskbin:
            taskbin_attrs = {'task_type': self.type.name}
            if self.Parameters.taskbin_release_type != 'none':
                taskbin_attrs['released'] = self.Parameters.taskbin_release_type
            self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
                state=ctr.State.READY,
                owner='BANNERLAND',
                attrs=taskbin_attrs,
            ).first().id
        else:
            self.Requirements.tasks_resource = self.Parameters.taskbin_resource

    def on_execute(self):
        attrs = {}
        add_call_args = ['--sandbox-task-id', str(self.id)]
        run_env = self.Parameters.run_env
        task_type = self.Parameters.bannerland_task_type
        make_banners_mode = self.Parameters.bannerland_make_banners_mode

        if task_type == 'perf':
            mode_env_key = 'BL_PERF_MODE'
            root_env_key = 'BL_PERF_ROOT'
            checked_attr_name = 'perf_preprod_checked'
        elif task_type == 'dyn':
            mode_env_key = 'BL_DYN_MODE'
            root_env_key = 'BL_DYN_ROOT'
            checked_attr_name = 'dyn_preprod_checked'
        elif task_type == 'recom':
            mode_env_key = 'BL_RECOM_MODE'
            root_env_key = 'BL_RECOM_ROOT'
            checked_attr_name = 'recom_preprod_checked'
        else:
            raise ValueError('task type "{}" not supported'.format(task_type))

        if run_env == 'preprod':
            os.environ['YT_POOL'] = 'bannerland-data'
            os.environ[mode_env_key] = 'preprod'
            add_call_args.append('--strict')
        elif run_env == 'prod':
            os.environ['YT_POOL'] = 'bannerland-prod'
            os.environ[mode_env_key] = 'prod'
            attrs[checked_attr_name] = True
        else:
            os.environ[root_env_key] = self.Parameters.yt_root

        if self.Parameters.maker_id:
            resource = BannerlandBinMakePocket.find(id=self.Parameters.maker_id).first()
        else:
            resource = BannerlandBinMakePocket.find(state='READY', attrs=attrs).order(-sdk2.Resource.id).first()
        resource_data = sdk2.ResourceData(resource)
        resource_path = resource_data.path
        resource_path_str = str(resource_path)

        os.environ['YT_TOKEN'] = self.Parameters.yt_token.value()
        os.environ['YQL_TOKEN'] = self.Parameters.yql_token.value()
        os.environ.setdefault('TMPDIR',  os.environ['TEMP'])

        cmd = ['./make_pocket', '--task-type', task_type, '--make-banners-mode', make_banners_mode] + add_call_args
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("make_pocket")) as pl:
            try:
                output = sp.check_output(cmd, stderr=sp.STDOUT, cwd=resource_path_str)
                pl.logger.info(output)
                if run_env == 'preprod' and 'make_banners OK!' in output:
                    pl.logger.info('make_banners bin resource is OK!')
                    setattr(resource, checked_attr_name, True)
                else:
                    pl.logger.info('make_banners did nothing')
            except sp.CalledProcessError as calledProcessError:
                pl.logger.error(calledProcessError.output)
                raise


class JunkOptozoraxResource(sdk2.Resource):
    release_subscribers = ['optozorax']
    pass


class JunkOptozoraxBlrtViewer(sdk2.Resource):
    release_subscribers = ['optozorax']
    pass


__all__ = (
    'BannerlandGenerationModels',
    'BannerlandBinMakePocket',
    'BannerlandRunMakePocket',

    'BannerLandWebBackendBinary',
    'BannerLandWebStatic',

    # this will be removed after tests
    'JunkOptozoraxResource',
    'JunkOptozoraxBlrtViewer',
)
