# -*- coding: utf-8 -*-

import os
import shutil

import sandbox.projects.release_machine.core.const as rm_const

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.irt.common import create_irt_data, do_bmgendict_copy

# Обновляемые SB-ресурсы в составе "generated_dicts" (sub_type -> массив копируемых файлов)
UPD_SB_RESOURCES = {
    "norm_dict": ["norm_dict"],
    "snorm_dict": ["word2snorm_dict"],
    "synonyms": ["syn_cells_ru", "syn_cells_tr"],
}

# Замороженные SB-ресурсы в составе "generated_dicts" (имя файла -> ID SB-ресурса)
FROZEN_SB_RESOURCES = {
    "csyns_translit_en2ru.smap": 1637291195,
    "csyns_translit_en2ru.strict.smap": 1637293519,
    "csyns_translit_ru2en.smap": 1637288549,
    "dict_directmod": 1637278408,
    "dict_directmod_full": 1637278968,
    "dict_equiv_domains": 1637271242,
    "mulword_syns_misspell": 1637284710,
}


class BmgenMakeGeneratedDicts(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Generate monolithic resource "generated_dicts" for Bannerland
    """

    class Parameters(sdk2.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        yt_proxy = sdk2.parameters.String(
            "YT proxy",
            default="arnold"
        )
        yt_token_vault_name = sdk2.parameters.String(
            "YT token vault name",
            default="yql_robot_bm_admin"
        )
        put_files_to_yt = sdk2.parameters.Bool(
            "Put neccessary dicts to YT",
            default=False
        )

    def on_execute(self):
        import irt.iron.options as iron_opts
        import yt.wrapper as yt
        from irt.bmgen.get_tragic_words import write_tragic_words
        from irt.sandbox import Sandbox
        from irt.utils import TemporaryDirectory, open_resource_or_file

        sb_client = Sandbox(sdk2.Vault.data("sandbox_oauth_token"))
        input_resources = sb_client.not_released_bmgen_resources(("freq_wordforms", "norm_dict", "wordcount_dict", "synonyms", "snorm_dict"))
        if input_resources is None:
            return

        if self.Parameters.binary_executor_release_type == rm_const.ReleaseStatus.stable:
            for sub_type in input_resources:
                sb_client.set_finite_ttl_for_previous_released_bmgen_resources(sub_type, released_type=rm_const.ReleaseStatus.stable)
            for sb_task in set(input_resources[x]['task']['id'] for x in input_resources):  # using `set` to prevent releasing same task twice
                self.server.release(task_id=sb_task, type=rm_const.ReleaseStatus.stable, subject='bmgen_release')

        work_norm_path = iron_opts.get("work_norm_path")
        gen_dicts_path = iron_opts.get("gen_dicts_path")

        with TemporaryDirectory() as tmp_dir:
            init_paths = dict()

            for src_name, copy_work_files in UPD_SB_RESOURCES.items():
                src_path = sb_client.download(input_resources[src_name]["id"], tmp_dir)
                for fname in copy_work_files:
                    init_paths[fname] = os.path.join(tmp_dir, src_path[0], work_norm_path, fname)

            for fname, frozen_src_id in FROZEN_SB_RESOURCES.items():
                init_paths[fname] = os.path.join(tmp_dir, sb_client.download(frozen_src_id, tmp_dir)[0])

            init_paths["dict_tragicword"] = os.path.join(tmp_dir, "dict_tragicword")
            write_tragic_words(sdk2.Vault.data("yql_robot_bm_admin"), init_paths["dict_tragicword"])

            sorted_fnames = sorted(init_paths)
            generated_dicts = create_irt_data(
                self,
                "generated_dicts",
                "Монолитный ресурс с основными словарями Bannerland-а",
                [os.path.join(gen_dicts_path, fname) for fname in sorted_fnames],
                ttl=30,
                check_task_mode=True,
            )
            for i in range(len(sorted_fnames)):
                shutil.copy(init_paths[sorted_fnames[i]], generated_dicts.filenames[i])
            do_bmgendict_copy(self, generated_dicts)

            if self.Parameters.put_files_to_yt:
                yt.config["proxy"]["url"] = self.Parameters.yt_proxy
                yt.config["token"] = sdk2.Vault.data(self.Parameters.yt_token_vault_name)
                yt_dicts = iron_opts.get("yt_files")["normalization_dicts"]
                with open(init_paths["norm_dict"]) as f_norm_dict, \
                        open(init_paths["word2snorm_dict"]) as f_snorm_dict, \
                        open_resource_or_file("dict_stwords_ru") as f_dict_stwords_ru:
                    yt.write_file(yt_dicts["norm_dict"], f_norm_dict)
                    yt.write_file(yt_dicts["snorm_dict"], f_snorm_dict)
                    yt.write_file(yt_dicts["stop_dict"], f_dict_stwords_ru)
