# -*- coding: utf-8 -*-

# manual: https://wiki.yandex-team.ru/irt/sandbox/#common
import os
from distutils.dir_util import copy_tree
from shutil import move
from sandbox import sdk2
from sandbox.projects.BM.gendict import BM_GENDICT
from sandbox.sdk2.resource import AbstractResource


class IrtData(AbstractResource):
    releasable = True
    auto_backup = True
    releasers = ['IRT']


# kwargs:
# ttl
# archive='tgz'
# any other names used as string attributes of resource: build_revision, etc...
def create_irt_data(task, sub_type, description, filenames, check_task_mode=False, **kwargs):
    tgz = kwargs.get('archive') == 'tgz'
    # NOTE: Fix this check and handle directly statuses of SB-resources (https://st.yandex-team.ru/IRT-2063)
    if (check_task_mode is True) and (task.Parameters.binary_executor_release_type != 'stable'):
        sub_type = '__{}__{}'.format(task.Parameters.binary_executor_release_type, sub_type)
    resource_root = sub_type + '.tgz' if tgz else sub_type
    result = sdk2.ResourceData(IrtData(task, description, resource_root, sub_type=sub_type, **kwargs))
    result.root_path = sdk2.path.Path(sub_type) if tgz else result.path
    result.root_path.mkdir(0o775, parents=True)
    result.tgz = tgz
    result.filenames = []
    for path in filenames:
        l = path.split('/')
        path_dirs = l[:-1]
        if path_dirs:
            result.root_path.joinpath(*path_dirs).mkdir(0o775, parents=True, exist_ok=True)
        result.filenames.append(str(result.root_path.joinpath(*l)))
    return result


def do_bmgendict_copy(task, irt_data_src):
    u"""
    Создает SB-ресурс BM_GENDICT как архивированную tgz-копию ресурса IRT_DATA (нужно для использования в BMYT и Qloud).

    :param task: экземпляр класса SB-таски, создающей ресурс
    :param irt_data_src: экземпляр класса ресурса IRT_DATA
    """
    sub_type = irt_data_src.root_path.name
    res_src = sdk2.ResourceData(BM_GENDICT(task, '', '{}.tgz'.format(sub_type), sub_type=sub_type, ttl=30))
    res_src.root_path = sdk2.path.Path(sub_type)

    init_rootpath_len = len(str(irt_data_src.root_path)) + 1
    res_src.filenames = [str(res_src.root_path.joinpath(x[init_rootpath_len:])) for x in irt_data_src.filenames]
    copy_tree(str(irt_data_src.root_path), str(res_src.root_path))

    res_src.tgz = True
    archive_irt_data(res_src)


def archive_irt_data(irt_data):
    if irt_data.tgz:
        tar_command = ['tar', 'czf', '-', '-C', str(irt_data.root_path)] + os.listdir(str(irt_data.root_path))
        with irt_data.path.open(mode='wb') as tgz_file:
            sdk2.helpers.subprocess.check_call(tar_command, stdout=tgz_file)


def build_irt_data(task, sub_type, description, dirname=None, **kwargs):
    tgz = kwargs.get('archive') == 'tgz'
    resource_root = sub_type + '.tgz' if tgz else sub_type
    build_dir = sdk2.path.Path(dirname if dirname else sub_type)
    irt_data = sdk2.ResourceData(IrtData(task, description, resource_root, sub_type=sub_type, **kwargs))
    if tgz:
        irt_data.tgz = True
        irt_data.root_path = build_dir
        archive_irt_data(irt_data)
    elif dirname:
        move(str(build_dir), str(irt_data.path))


class YTParameters(sdk2.Parameters):
    yt_proxy = sdk2.parameters.String(
        "YT proxy",
        default="hahn"
    )
    yt_pool = sdk2.parameters.String(
        "YT pool",
        default="catalogia"
    )
    yt_token_vault_name = sdk2.parameters.String(
        "YT token vault name",
        default="yql_robot_bm_admin"
    )
