# coding: utf-8
import shutil
import sandbox.common.types.task as ctt

from sandbox import common
from sandbox import sdk2
from sandbox.projects.iss.import_yp_microservices import VersionParameter
from sandbox.projects.iss import QEMU_IMAGE_ISS_LOCAL, LXC_IMAGE_ISS_LOCAL
from sandbox.projects.sandbox.sandbox_lxc_image import SandboxLxcImage, UbuntuRelease
from sandbox.projects.sandbox.resources import LXC_CONTAINER


RESOURCE_TTL = 14
UBUNTU_RELEASES = (
    ("lucid", "lucid"),
    ("precise", "precise"),
    ("trusty", "trusty"),
    ("xenial", "xenial"),
    ("bionic", "bionic")
)


LXC_SCRIPT = """
set -eux
set -o pipefail

export DEBIAN_FRONTEND="noninteractive"
APT_GET="apt-get --yes --no-install-recommends"

umask 0022
apt-get update

echo "root:root" | chpasswd

$APT_GET install socat
$APT_GET install \
  qemu-static={qemu_static_version}
$APT_GET install libguestfs-tools

# cleanup apt
apt-get clean
find var/cache/apt/archives -iname '*.deb' -delete
find var/lib/apt/lists -type f -delete
find var/cache/apt -iname '*.bin' -delete
"""


class BuildQemuImageIssLocal(sdk2.Task):
    """
    Task for building qemu image with embedded iss-agent ready for integrational tests
    """

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Environment'):
            iss_version = VersionParameter(
                spec='ru.yandex.iss:iss-agent-uber-jar',
                description='iss-agent version',
                required=True
            )
            kernel_version = sdk2.parameters.String(
                'linux kernel version',
                default='4.14.55-18',
                required=True
            )
            porto_version = sdk2.parameters.String(
                'porto version',
                default='4.18.9',
                required=True
            )
            iss_setup_script = sdk2.parameters.String(
                'iss-agent setup script',
                default='arcadia:/robots/trunk/genconf/PORTOVM/iss_local/qemu-iss-agent.sh',
                required=True
            )
            ubuntu_release = sdk2.parameters.String(
                'ubuntu release',
                default='xenial',
                choices=UBUNTU_RELEASES,
                required=True
            )

    def on_execute(self):
        qemu_image_id = self.Context.qemu_image_id
        ubuntu_release = self.Parameters.ubuntu_release
        if not qemu_image_id:
            with self.memoize_stage.create_subtasks:
                decription = 'Quemu image for iss-agent {version}'.format(version=self.Parameters.iss_version)
                output_image_id = QEMU_IMAGE_ISS_LOCAL(
                    self,
                    decription,
                    'iss_local.img',
                    ttl=RESOURCE_TTL
                ).id

                build_layer_task = sdk2.Task['BUILD_PORTO_LAYER'](
                    self,
                    description=decription,
                    output_resource_id=output_image_id,
                    compress='tar.xz',
                    layer_name='iss_local',
                    layer_type='QEMU_IMAGE_ISS_LOCAL',
                    bootstrap='\n'.join([
                        'arcadia:/robots/trunk/genconf/PORTOVM/common/bootstrap_ubuntu_%s.sh' % ubuntu_release,
                    ]),
                    script='\n'.join([
                        'arcadia:/robots/trunk/genconf/PORTOVM/common/ubuntu_%s_settings.sh' % ubuntu_release,
                        'arcadia:/robots/trunk/genconf/PORTOVM/common/yandex-apt.sh',
                        'arcadia:/robots/trunk/genconf/PORTOVM/kernel/cloud-init.sh',
                        'arcadia:/robots/trunk/genconf/PORTOVM/kernel/install-kernel.sh',
                        'arcadia:/robots/trunk/genconf/PORTOVM/kernel/install-extlinux.sh',
                        self.Parameters.iss_setup_script,
                        'arcadia:/robots/trunk/genconf/PORTOVM/common/apt-cleanup.sh',
                    ]),
                    script2='\n'.join([
                        'arcadia:/robots/trunk/genconf/PORTOVM/common/cleanup.sh',
                    ]),
                    script_env={
                        'KERNEL_VERSION': self.Parameters.kernel_version,
                        'PORTO_VERSION': self.Parameters.porto_version,
                        'ISS_VERSION': self.Parameters.iss_version,
                        'virt_mode': 'qemu',
                    },
                    start_os=True,
                    keep_resolv_conf=True,
                    merge_layers=True,
                ).enqueue()

                self.Context.qemu_image_id = output_image_id
                raise sdk2.WaitTask(
                    [build_layer_task],
                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                )
        elif not sdk2.Resource.find(id=qemu_image_id).first():
            raise common.errors.TaskFailure("QEMU_IMAGE_ISS_LOCAL with id=%s is not ready" % qemu_image_id)
        else:
            self.Parameters.tags = [
                "iss-agent-%s" % self.Parameters.iss_version,
                "porto-%s" % self.Parameters.porto_version,
                "kernel-%s" % self.Parameters.kernel_version,
                ubuntu_release,
            ]


class BuildLxcContainerIssLocal(sdk2.Task):
    """
    Task for building LXC container image for running tests using QEMU_IMAGE_ISS_LOCAL
    """

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Environment'):
            ubuntu_release = SandboxLxcImage.Parameters.ubuntu_release(default=UbuntuRelease.XENIAL)
            qemu_static_version = sdk2.parameters.String(
                'qemu-static version',
                default='2.10.2',
                required=True
            )

    def build_image(self):
        params = {
            SandboxLxcImage.Parameters.ubuntu_release.name: self.Parameters.ubuntu_release,
            SandboxLxcImage.Parameters.resource_description.name: 'LXC container image for running tests using QEMU_IMAGE_ISS_LOCAL',
            SandboxLxcImage.Parameters.resource_type.name: LXC_CONTAINER.name,
            SandboxLxcImage.Parameters.custom_script.name: LXC_SCRIPT.format(qemu_static_version=self.Parameters.qemu_static_version),
            SandboxLxcImage.Parameters.custom_image.name: True,
            SandboxLxcImage.Parameters.test_result_lxc.name: True,
            SandboxLxcImage.Parameters.install_common.name: True,
        }

        build_task = SandboxLxcImage(self, **params)

        self.Context.build_task_id = build_task.enqueue().id
        raise sdk2.WaitTask(
            self.Context.build_task_id,
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
        )

    def on_execute(self):
        build_task_id = self.Context.build_task_id
        if not build_task_id:
            with self.memoize_stage.create_subtasks:
                self.build_image()
        else:
            build_task = sdk2.Task[self.Context.build_task_id]
            if build_task.status != ctt.Status.SUCCESS:
                raise common.errors.TaskFailure("LXC image build failue")

            resource = sdk2.Resource\
              .find(task_id=build_task_id, resource_type=LXC_CONTAINER)\
              .first()

            data = sdk2.ResourceData(resource)

            destination_path = 'rootfs.tar.gz'
            shutil.copy(str(data.path), destination_path)

            resource_meta = LXC_IMAGE_ISS_LOCAL(
                self,
                'LXC image for iss-local with quemu-static={version}'.format(version=self.Parameters.qemu_static_version),
                destination_path,
                ttl=RESOURCE_TTL,
                platform=resource.platform
            )

            resource = sdk2.ResourceData(resource_meta)
            resource.ready()

            self.Parameters.tags = [
                "qemu-static-%s" % self.Parameters.qemu_static_version,
                self.Parameters.ubuntu_release,
            ]
