# coding: utf-8
import sandbox.common.types.task as ctt

from sandbox import common
from sandbox import sdk2
from sandbox.projects.iss import QEMU_IMAGE_ISS_TEAMCITY
import logging


RESOURCE_TTL = 14
UBUNTU_RELEASES = (
    ("lucid", "lucid"),
    ("precise", "precise"),
    ("trusty", "trusty"),
    ("xenial", "xenial"),
    ("bionic", "bionic")
)
BOOTSTRAP_UBUNTU_TEMPLATE = \
    "arcadia:/robots/trunk/genconf/PORTOVM/common/bootstrap_ubuntu_%s.sh"

UBUNTU_SETTINGS_TEMPLATE = \
    "arcadia:/robots/trunk/genconf/PORTOVM/common/ubuntu_%s_settings.sh"

DEFAULT_SCRIPTS_LIST = [
    'arcadia:/robots/trunk/genconf/PORTOVM/common/yandex-apt.sh',
    'arcadia:/robots/trunk/genconf/PORTOVM/kernel/cloud-init.sh',
    'arcadia:/robots/trunk/genconf/PORTOVM/kernel/install-kernel.sh',
    'arcadia:/robots/trunk/genconf/PORTOVM/kernel/install-extlinux.sh',
    'arcadia:/robots/trunk/genconf/PORTOVM/common/apt-cleanup.sh',
]

DEFAULT_CLEANUP_SCRIPT = \
    'arcadia:/robots/trunk/genconf/PORTOVM/common/cleanup.sh'

DEFAULT_NOT_READY_MESSAGE = \
    "QEMU_IMAGE_ISS_TEAMCITY with id=%s is not ready"


class IssQemuImageTeamcity(sdk2.Task):
    """
    Task for building qemu image for ISS teamcity
    """

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Environment'):
            kernel_version = sdk2.parameters.String(
                'linux kernel version',
                default='4.14.55-18',
                required=True
            )
            porto_version = sdk2.parameters.String(
                'porto version',
                default='4.18.9',
                required=True
            )
            disk_manager_version = sdk2.parameters.String(
                'diskmanager version',
                default='4.18.9',
                required=True
            )
            ubuntu_release = sdk2.parameters.String(
                'ubuntu release',
                default='xenial',
                choices=UBUNTU_RELEASES,
                required=True
            )

    def on_execute(self):
        qemu_image_id = self.Context.qemu_image_id
        ubuntu_release = self.Parameters.ubuntu_release
        logging.info("On execution. qemu image id: %s. ubuntu release: %s")
        if not qemu_image_id:
            logging.info("no qemu image id. Start porto task")
            with self.memoize_stage.create_subtasks:
                output_image_id = QEMU_IMAGE_ISS_TEAMCITY(
                    self,
                    'Quemu image for teamcity {version}'.format(version=3),
                    'iss_teamcity.img',
                    ttl=RESOURCE_TTL
                ).id

                dm_version = self.Parameters.disk_manager_version

                build_layer_task = sdk2.Task['BUILD_PORTO_LAYER'](
                    self,
                    description="QEMU image for iss teamcity",
                    output_resource_id=output_image_id,
                    compress='tar.xz',
                    layer_name='iss_teamcity',
                    layer_type='QEMU_IMAGE_ISS_TEAMCITY',
                    bootstrap='\n'.join([
                        BOOTSTRAP_UBUNTU_TEMPLATE % ubuntu_release,
                    ]),
                    script='\n'.join(
                        [UBUNTU_SETTINGS_TEMPLATE % ubuntu_release] +
                        DEFAULT_SCRIPTS_LIST
                    ),
                    script2='\n'.join([DEFAULT_CLEANUP_SCRIPT]),
                    script_env={
                        'KERNEL_VERSION': self.Parameters.kernel_version,
                        'PORTO_VERSION': self.Parameters.porto_version,
                        'DISK_MANAGER_VERSION': dm_version,
                        'virt_mode': 'qemu',
                    },
                    start_os=True,
                    keep_resolv_conf=True,
                    merge_layers=True,
                ).enqueue()

                self.Context.qemu_image_id = output_image_id
                raise sdk2.WaitTask(
                    [build_layer_task],
                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                )
        elif not sdk2.Resource.find(id=qemu_image_id).first():
            message = DEFAULT_NOT_READY_MESSAGE % qemu_image_id
            raise common.errors.TaskFailure(message)
        else:
            self.Parameters.tags = [
                "porto-%s" % self.Parameters.porto_version,
                "kernel-%s" % self.Parameters.kernel_version,
                ubuntu_release,
            ]
