# coding: utf-8
import logging
import os
import subprocess

from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.resource_types import iss_releasers
from sandbox.projects.iss import (
    CACHER_APPLICATION_CONF_TEMPLATE,
    CACHER_JOLOKIA_POLICY,
    CACHER_JOLOKIA_PROPERTIES,
    CACHER_LOGBACK_XML,
    CACHER_LOOP_CONF,
    CACHER_STATUS_SCRIPT_TEMPLATE
)


class GitTag(SandboxStringParameter):
    name = 'git_tag'
    description = 'Git tag or branch name'
    default_value = 'develop'
    required = True


class ImportCacherHooksAndConfigs(SandboxTask):
    """
        Import ISS cacher hooks and options from Git and make Sandbox resource from them
        TODO: refactor with JAPI task
    """

    GIT_URL = "ssh://git@bb.yandex-team.ru/search_infra/schedulers-deploy.git"
    CHECKOUT_DIR = "working_copy"
    VAULT_OWNER = 'ISS'
    SSH_KEY_VAULT_NAME = 'robot-verter-secret'
    type = 'IMPORT_CACHER_HOOKS_AND_CONFIGS'
    release_to = iss_releasers
    input_parameters = [
        GitTag
    ]

    def _get_checkout_dir(self):
        return self.abs_path(self.CHECKOUT_DIR)

    def do_checkout(self):
        checkout_path = self._get_checkout_dir()
        tag = self.ctx.get('git_tag')
        assert tag, 'Trying to fetch project from git, but no tag specified'

        with ssh.Key(self, self.VAULT_OWNER, self.SSH_KEY_VAULT_NAME):
            run_process(
                ['git', 'clone', self.GIT_URL, checkout_path],
                log_prefix='git_clone',
                shell=True,
            )
            run_process(
                ['git', 'checkout', tag],
                work_dir=checkout_path,
                log_prefix='git_checkout',
                shell=True)

            p = run_process(
                ["git", "rev-parse", "--short", "HEAD"],
                work_dir=checkout_path,
                log_prefix='get HEAD rev',
                stdout=subprocess.PIPE,
                shell=True)

        # yes, I know - define fields outside of init
        self._checkout_rev = p.communicate()[0].splitlines()[0]

    def _get_checkout_rev(self):
        return self._checkout_rev

    def __create_one_resource(self, resource_type, relative_path):
        description = '%s @ %s' % \
                      (relative_path, self._get_checkout_rev())
        logging.info('create resource: %s' % (description,))
        self.create_resource(
            description=description,
            resource_path=os.path.join(self.CHECKOUT_DIR, relative_path),
            resource_type=resource_type,
            attributes={
                'ttl': 365,
            }
        )

    def do_create_resources(self):
        self.__create_one_resource(CACHER_APPLICATION_CONF_TEMPLATE, "cacher_hooks/application.conf.template")
        self.__create_one_resource(CACHER_JOLOKIA_POLICY, "cacher_hooks/jolokia.policy")
        self.__create_one_resource(CACHER_JOLOKIA_PROPERTIES, "cacher_hooks/jolokia.properties")
        self.__create_one_resource(CACHER_LOGBACK_XML, "cacher_hooks/logback.xml")
        self.__create_one_resource(CACHER_LOOP_CONF, "cacher_hooks/loop.conf")
        self.__create_one_resource(CACHER_STATUS_SCRIPT_TEMPLATE, "cacher_hooks/status.sh.template")

    def on_execute(self):
        self.do_checkout()
        self.do_create_resources()

    def arcadia_info(self):
        return None, None, self.ctx.get('git_tag')


__Task__ = ImportCacherHooksAndConfigs
