# coding: utf-8
import cStringIO
import urllib2
import logging
import zipfile

from sandbox import sdk2
from sandbox.projects.common.nanny import nanny
from sandbox.common.utils import singleton_classproperty
from xml.etree import ElementTree
from contextlib import closing

import sandbox.projects.iss


RESOURCE_TTL = 14
ARTIFACTORY_TIMEOUT_SECONDS = 30
ARTIFACT_REPO = 'http://artifactory.yandex.net/artifactory/qe'


class ArtifactTemplate(object):
    def __init__(self, group_id, artifact_id, packaging='jar', classifier=None):
        self.group_id = group_id
        self.artifact_id = artifact_id
        self.packaging = packaging
        self.classifier = classifier

    def basename(self, version):
        classifier_postfix = ''
        if self.classifier:
            classifier_postfix = '-' + self.classifier
        return self.artifact_id + '-' \
               + version + classifier_postfix \
               + '.' + self.packaging

    def url(self, base_url, version):
        artifact_basename = self.basename(version)
        group_id_path = self.group_id.replace('.', '/')
        artifact_id_path = self.artifact_id.replace('.', '/')
        return '/'.join([
            base_url,
            group_id_path,
            artifact_id_path,
            version,
            artifact_basename
        ])


def download_maven_artifact(artifact_spec, version):
    paths = artifact_spec.split('!/')

    archive_path = None
    if len(paths) == 2:
        artifact_spec, archive_path = paths

    tmpl = ArtifactTemplate(*artifact_spec.split(':'))
    artifact_url = tmpl.url(ARTIFACT_REPO, version)

    resource_path = tmpl.artifact_id + '.' + tmpl.packaging
    if archive_path:
        resource_path = archive_path

    logging.info("artifact url: {}, resource path: {}".format(artifact_url, resource_path))

    with open(resource_path, 'w') as resource_file:
        with closing(urllib2.urlopen(artifact_url, timeout=ARTIFACTORY_TIMEOUT_SECONDS)) as artifact_file:
            if archive_path:
                with zipfile.ZipFile(cStringIO.StringIO(artifact_file.read())) as archive:
                    resource_file.write(archive.read(archive_path))
            else:
                resource_file.write(artifact_file.read())
        return resource_path


class VersionParameter(sdk2.parameters.String):
    name = 'version'
    description = 'YP microservices artifacts version'
    required = True
    spec = None

    @singleton_classproperty
    def metadata_tree(cls):
        xml_url = '/'.join([
            ARTIFACT_REPO,
            cls.spec.replace(':', '/').replace('.', '/'),
            'maven-metadata.xml'
        ])

        metadata_xml = urllib2.urlopen(xml_url, timeout=ARTIFACTORY_TIMEOUT_SECONDS).read()
        return ElementTree.fromstring(metadata_xml)

    @singleton_classproperty
    def choices(cls):
        try:
            versions = [element.text for element in cls.metadata_tree.find('versioning').find('versions')]
            return [(x, x) for x in reversed(versions)]
        except:
            pass

    @singleton_classproperty
    def default_value(cls):
        try:
            return cls.metadata_tree.find('versioning').find('release').text
        except Exception as e:
            logging.info("Exception retrieving maven metadata: {}".format(e))
            return "Could not retrieve from artifactory, refresh or enter manually"


class ImportYpMicroservicesBundle(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    A task which imports yp microservices binaries/scripts and configs from artifactory
    """

    class Requirements(sdk2.Requirements):
        disk_space = 2048

    class Parameters(sdk2.Task.Parameters):
        version = VersionParameter(spec='ru.yandex.yp:yp-services-parent')

    def _make_resource(self, artifact_spec, resource_type):
        version = self.Parameters.version
        resource_path = download_maven_artifact(artifact_spec, version)
        resource_meta = resource_type(self, artifact_spec + ":" + version, resource_path, version=version, ttl=RESOURCE_TTL)
        resource = sdk2.ResourceData(resource_meta)
        resource.ready()

    def _create_resources(self):
        # scripts
        self._make_resource("ru.yandex.yp:deploy-scripts:zip:archive!/create_tmpfs.py", sandbox.projects.iss.YP_CREATE_TMPFS_SCRIPT)
        self._make_resource("ru.yandex.yp:deploy-scripts:zip:archive!/jdk_mux.sh", sandbox.projects.iss.YP_JDK_MUX_SCRIPT)
        self._make_resource("ru.yandex.yp:deploy-scripts:zip:archive!/oom_killer.sh", sandbox.projects.iss.YP_OOM_KILLER_SCRIPT)
        self._make_resource("ru.yandex.yp:deploy-scripts:zip:archive!/yasm_stw_unistat.py", sandbox.projects.iss.YP_YASM_STW_UNISTAT_SCRIPT)

        # services
        self._make_resource("ru.yandex.yp.export:yp-export-service", sandbox.projects.iss.YP_EXPORT_SERVICE_JAR)
        self._make_resource("ru.yandex.yp.hfsm:yp-hfsm-service", sandbox.projects.iss.YP_HFSM_SERVICE_JAR)
        self._make_resource("ru.yandex.yp.hfsm:qloud-hfsm-service", sandbox.projects.iss.QLOUD_HFSM_SERVICE_JAR)
        self._make_resource("ru.yandex.yp.pdns:yp-pdns-backend", sandbox.projects.iss.YP_PDNS_BACKEND_JAR)
        self._make_resource("ru.yandex.yp.export:yp-cauth-export-service", sandbox.projects.iss.YP_CAUTH_EXPORT_SERVICE_JAR)

    def on_execute(self):
        self._create_resources()
