# -*- coding: utf-8 -*-
import logging
import os
import time

from sandbox.projects.common import error_handlers as eh
import sandbox.projects.release_machine.mixins.build as rm_build_mixin
import sandbox.projects.common.arcadia.sdk as sdk
import sandbox.projects.common.build.parameters as params
import sandbox.projects.common.nanny.nanny as nanny
import sandbox.projects.common.vcs.aapi as aapi
import sandbox.sandboxsdk.paths as paths
import sandbox.sdk2 as sdk2

STRING = 'string'
RESOURCE = 'resource'
LOCAL_RELEASE_DIR = 'release'


class RecommenderManagerExecutable(sdk2.resource.Resource):
    """
        Recommender Manager Binary
    """
    releasable = True
    any_arch = False
    executable = True
    auto_backup = True
    releasers = ["ITDITP", "robot-srch-releaser"]
    arcadia_build_path = 'quality/trailer/recommender/manager/recommender_manager'
    arcadia_build_name = 'recommender_manager'


BUILD_FIELDS = {
    'recommender_manager': RecommenderManagerExecutable,
}


class BuildRecommenderManager(rm_build_mixin.ComponentReleaseTemplate, nanny.ReleaseToNannyTask2, sdk2.Task):
    """
        Build binaries for itditp recommender manager releases with arcadia sdk
    """
    class Requirements(sdk2.Task.Requirements):
        disk_space = 80 * 1024

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group(params.BASE_BUILD_GROUP_NAME) as bb_group:
            checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia', required=True)
            clear_build = sdk2.parameters.Bool('Clear build')
            use_fuse = sdk2.parameters.Bool("Use fuse to checkout arcadia", default=True)
        with sdk2.parameters.Group(params.BUILD_SYSTEM_GROUP_NAME) as bs_group:
            with sdk2.parameters.String('Build type', required=True) as build_type:
                build_type.values.release = build_type.Value('Release', default=True)
                build_type.values.debug = 'Debug'
                build_type.values.profile = 'Profile'
                build_type.values.coverage = 'Coverage'
                build_type.values.relwithdebinfo = 'Release with debug info'
                build_type.values.valgrind = 'Valgrind (debug)'
                build_type.values.valgrind_release = 'Valgrind (release) (only for build with ya)'

            with sdk2.parameters.String("Build system") as build_system:
                build_system.values.ya = "Ya"
                build_system.values.ya_force = "Ya force"
                build_system.values.semi_distbuild = build_system.Value("Semi distbuild", default=True)
                build_system.values.distbuild = "distbuild"

            definition_flags = sdk2.parameters.String(
                'Definition flags (for example "-Dkey1=val1 ... -DkeyN=valN")', required=False
            )

            target_platform_flags = sdk2.parameters.String(
                'Target platform flags (only for cross-compilation)', required=False
            )
            thinlto = sdk2.parameters.Bool("Build with ThinLTO", default=False)

    def on_enqueue(self):
        for ft in self.get_targets():
            ft_name = getattr(BUILD_FIELDS[ft], 'arcadia_build_name', BUILD_FIELDS[ft].arcadia_build_path)
            t_res = BUILD_FIELDS[ft](
                self, ft_name, os.path.join(LOCAL_RELEASE_DIR, BUILD_FIELDS[ft].arcadia_build_path)
            )
            setattr(self.Context, "{}_res_id".format(ft), t_res.id)

    def get_targets(self):
        return BUILD_FIELDS.keys()

    def on_execute(self):
        arcadia_src = self.Parameters.checkout_arcadia_from_url
        self.Context.checkout_arcadia_from_url = arcadia_src.strip("/")

        target_dirs = [os.path.dirname(BUILD_FIELDS[rk].arcadia_build_path) for rk in self.get_targets()]
        sdk2.path.Path(os.path.abspath(LOCAL_RELEASE_DIR)).mkdir(mode=0o755, exist_ok=True)

        def_flags = sdk.parse_flags(self.Parameters.definition_flags)
        def_flags['SANDBOX_TASK_ID'] = self.id

        if self.Parameters.use_fuse:
            self._fuse_build(arcadia_src, def_flags, target_dirs)
        else:
            self._build(arcadia_src, def_flags, target_dirs)

    def _do_build(self, arcadia_src, target_dirs, def_flags, checkout):
        build_start = int(time.time())
        sdk.do_build(
            self.Parameters.build_system,
            arcadia_src,
            target_dirs,
            self.Parameters.build_type,
            clear_build=self.Parameters.clear_build,
            results_dir=LOCAL_RELEASE_DIR,
            def_flags=def_flags,
            target_platform_flags=self.Parameters.target_platform_flags,
            checkout=checkout,
            thinlto=self.Parameters.thinlto,
        )
        self._time_info("build_time", int(time.time()) - build_start)

    def _fuse_build(self, arcadia_src, def_flags, target_dirs):
        checkout_start = int(time.time())
        path, revision = aapi.ArcadiaApi.extract_path_and_revision(arcadia_src)
        eh.ensure(path, "Url {} is not present in arcadia-api".format(arcadia_src))

        if revision is None:
            revision = aapi.ArcadiaApi.svn_head()

        if not aapi.ArcadiaApi.has_svn_revision(revision):
            eh.ensure(
                aapi.ArcadiaApi.wait_svn_revision(revision, 180),
                "Revision {} is not present in arcadia-api. Waited 180s.".format(revision)
            )
        logging.info("Release revision is %s.", revision)

        with aapi.ArcadiaApi.mount_svn_path(path, revision=revision) as source_dir_origin:
            logging.info("Arcadia source dir: %s", source_dir_origin)

            arc_dir = paths.make_folder("arc")
            source_dir = os.path.join(arc_dir, 'arcadia')
            self._ensure_link(source_dir_origin, source_dir)
            self._time_info("checkout_time", int(time.time()) - checkout_start)
            self._do_build(source_dir, target_dirs, def_flags, False)

    def _build(self, arcadia_src, def_flags, target_dirs):
        # Checkout usefull part of Arcadia to decrease checkout time
        do_clone_start = int(time.time())
        arcadia_src_dir = sdk.do_clone(arcadia_src, self, use_checkout=False)
        self._time_info("clone_time", int(time.time()) - do_clone_start)
        checkout_start = int(time.time())
        release_svn_info = sdk2.svn.Arcadia.info(arcadia_src)
        logging.info("Release revision is %s.", release_svn_info["entry_revision"])
        for dependent_part in [
            "contrib",
            "dict",
            "geobase",
            "kernel",
            "library",
            "mapreduce",
            "quality",
            "search",
            "tools",
            "util",
            "web",
            "yql",
            "ysite",
            "yweb",
        ]:
            # use only top level directories
            sdk2.svn.Arcadia.update(
                os.path.join(arcadia_src_dir, dependent_part),
                revision=release_svn_info["entry_revision"],
                set_depth="infinity"
            )

        self._time_info("checkout_time", int(time.time()) - checkout_start)
        self._do_build(arcadia_src_dir, target_dirs, def_flags, True)

    def _time_info(self, time_ctx, time_val):
        setattr(self.Context, time_ctx, time_val)
        self.set_info("{0} is {1} sec.".format(time_ctx, time_val))

    @staticmethod
    def _ensure_link(path, link):
        if os.path.exists(link):
            eh.ensure(os.path.islink(link), "Cannot create symlink, non-link already exists: {}".format(link))
            os.remove(link)
        os.symlink(path, link)
