# -*- coding: utf-8 -*-

import aniso8601
import json
import logging
import socket
import subprocess
import time
from urlparse import urlparse

from sandbox import sdk2
from sandbox import common
from sandbox.projects import resource_types
import sandbox.projects.common.search.requester as requester
import sandbox.projects.itditp.BuildRecommenderManager as itditp_rm
import sandbox.projects.itditp.resources as itditp_resources
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.common.utils import gimme_port


class RecommenderManagerInstance(object):
    """
        Recommender manager instance initialised by RecommenderManagerExecutable
        resource. If one is None production recommender_manager is used
    """
    def __init__(self, manager_resource=None, port="15627"):
        if (manager_resource is not None):
            self.manager_path = str(sdk2.ResourceData(manager_resource).path)
        else:
            self.manager_path = None
        self.rm_process = None
        self.port = port

    def start(self):
        logging.info("Instantiating recommender manager")
        if (self.manager_path is not None):
            recommender_manager_cmd = [
                self.manager_path,
                "--port", str(self.port),
                "--log-file-path", "recommender.log"
            ]
            self.rm_process = subprocess.Popen(recommender_manager_cmd)
            time.sleep(20)  # Waiting 20 seconds for manager to bootstrap
        logging.info("Done instantiating recommender manager")

    def stop(self):
        if (self.rm_process is not None):
            pass
            self.rm_process.terminate()
        logging.info("Terminated recommender manager")

    def get_augment_url(self):
        if (self.rm_process is None):
            return "&waitall=da&timeout=10000000&reqinfo=dolbilo"
        else:
            return "&waitall=da&timeout=10000000&reqinfo=dolbilo&srcrwr=RECOMMENDER_MANAGER:{}:{}:100000000&srcrwr=RECOMMENDER_SETUP:{}:{}:100000000"\
                .format(socket.gethostname(), self.port, socket.gethostname(), self.port)


class RecommenderManagerRequester(requester.Requester):
    """
        Makes given by requests_iterator requests and saves json resonse into self.response filed
    """
    def __init__(self, request_url="https://hamster.yandex.ru", augmenturl="&reqinfo=dolbilo"):
        super(RecommenderManagerRequester, self).__init__()
        self.responses = dict()
        self.re_try_limit = 15

        self.request_url = request_url
        self.augmenturl = augmenturl
        self.timeout = 100000000

    def build_request(self, req):
        return self.request_url + req + self.augmenturl

    def on_response(self, nreq, resultJson):
        try:
            result = json.loads(resultJson)
            self.responses[nreq] = result
        except ValueError:
            logging.warning("json.load except ValueError, json is bad")


class CompareRecommenderManagerResponses(sdk2.Task):
    """
        Makes requests to two given recommender managers, ensures that all responses do not differ
    """
    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment("jsonpath_rw"),
            PipEnvironment("yandex-yt"),
            PipEnvironment("yandex-yt-yson-bindings-skynet"),
        ]

    class Parameters(sdk2.Parameters):
        yt_token = sdk2.parameters.Vault("YT token vault (for fetching log)", required=True)

        with sdk2.parameters.Group("Compare responses parameters") as itditp_compare_params_group:
            requests_num = sdk2.parameters.Integer("Number of requests", default=100)
            requests_plan = sdk2.parameters.Resource(
                "Requests (will fetch arbitrary if not specified)",
                resource_type=itditp_resources.ITDITP_REQUESTS_RECOMMENDER
            )
            json_filter = sdk2.parameters.String("json-rw filter for diff checking", default="data[0].construct.recommendations.[*].url")

        with sdk2.parameters.Group("Recommender manager parameters 1") as itditp_recommender_params_1:
            manager1_executable = sdk2.parameters.Resource(
                "Recommender manager executable (use production if not specified)",
                resource_type=itditp_rm.RecommenderManagerExecutable
            )

        with sdk2.parameters.Group("Recommender manager parameters 2") as itditp_recommender_params_2:
            manager2_executable = sdk2.parameters.Resource(
                "Recommender manager executable (use production if not specified)",
                resource_type=itditp_rm.RecommenderManagerExecutable
            )

    def _get_requests(self):
        """
            Fetch production requests with uuid and icoockie from YT logs at
            arnold.`logs/recommender-reqans-log/30min`
        """
        YT_LOGS_PATH = "//logs/recommender-reqans-log/30min"

        requests = []
        if (self.Parameters.requests_plan is None):
            import yt.wrapper as yt
            from yt.yson import yson_to_json

            yt_client = yt.YtClient(proxy="arnold", token=self.Parameters.yt_token.data())
            log_dir = yt_client.get(YT_LOGS_PATH)
            table = yt.ypath_join(YT_LOGS_PATH, max(log_dir.keys(), key=aniso8601.parse_datetime))
            for node in yt_client.read_table(yt.TablePath(table, columns=["userId", "fullRequest"], start_index=0, end_index=self.Parameters.requests_num)):
                node = yson_to_json(node)
                request = urlparse(node["fullRequest"])._replace(netloc="", scheme="").geturl()  # https://yandex.ru/search?query -> /search?query
                if ("userId" in node):
                    if ("iCookie" in node["userId"]):
                        request += "&icookie=" + node["userId"]["iCookie"]
                    if ("yandexUid" in node["userId"]):
                        request += "&uuid=" + node["userId"]["yandexUid"]
                        request += "&yandexuid=" + node["userId"]["yandexUid"]
                requests.append(request)
        else:
            with open(str(sdk2.ResourceData(self.Parameters.requests_plan).path, "r")) as plan:
                while (True):
                    request = plan.readline()
                    if (not request) or (len(requests) >= self.Parameters.requests_num):
                        break
                    requests.append(request)
        if ((len(requests) > 0) and (len(requests) < self.Parameters.requests_num)):
            requests = requests + requests * (self.Parameters.requests_num // len(requests))
        return requests[:self.Parameters.requests_num]

    def on_execute(self):
        import jsonpath_rw

        requests_plan = self._get_requests()

        self.manager1 = RecommenderManagerInstance(self.Parameters.manager1_executable, port=gimme_port())
        self.manager1.start()

        self.manager2 = RecommenderManagerInstance(self.Parameters.manager2_executable, port=gimme_port())
        self.manager2.start()

        manager1_requester = RecommenderManagerRequester(augmenturl=self.manager1.get_augment_url())
        manager2_requester = RecommenderManagerRequester(augmenturl=self.manager2.get_augment_url())

        manager1_requester.init(requests_plan)
        manager2_requester.init(requests_plan)

        logging.info("Started sending requests to managers")
        manager1_requester.send_requests()
        manager2_requester.send_requests()
        logging.info("Finished sending requests to managers")

        jsonpath_expr = jsonpath_rw.parse(self.Parameters.json_filter)  # json-rw expression, used to extract fields that shall not differ
        for nreq in range(self.Parameters.requests_num):
            request = requests_plan[nreq]
            if (nreq not in manager1_requester.responses):
                raise common.errors.TaskFailure("Failed to get response for request \"{}\" on manager1".format(request))
            if (nreq not in manager2_requester.responses):
                raise common.errors.TaskFailure("Failed to get response for request \"{}\" on manager2".format(request))

            response1 = manager1_requester.responses[nreq]
            response2 = manager2_requester.responses[nreq]

            response1_fields = [str(i.value) for i in jsonpath_expr.find(response1)]
            response2_fields = [str(i.value) for i in jsonpath_expr.find(response2)]

            if (response1_fields != response2_fields):
                with open("manager1_response.json", "w") as r1, open("manager2_response.json", "w") as r2:
                    json.dump(manager1_requester.responses[nreq], r1)
                    json.dump(manager2_requester.responses[nreq], r2)
                sdk2.ResourceData(resource_types.PLAIN_TEXT(
                    self, "Manager 1 response", "manager1_response.json"
                )).ready()
                sdk2.ResourceData(resource_types.PLAIN_TEXT(
                    self, "Manager 2 response",  "manager2_response.json"
                )).ready()
                raise common.errors.TaskFailure("On request {} (#{}) responses differed. See resources: {}. Responses: {}".format(
                    request,
                    nreq,
                    list(map(lambda r: r.url, sdk2.Resource.find(resource_types.PLAIN_TEXT, task=self).limit(2))),
                    [response1_fields, response2_fields]
                ))

    def on_finish(self, prev_status, status):
        if (hasattr(self, "manager1")):
            self.manager1.stop()
        if (hasattr(self, "manager2")):
            self.manager2.stop()
        return super(self.__class__, self).on_finish(prev_status, status)
