# -*- coding: utf-8 -*-

# Ideologically this is a clone of GET_BASESEARCH_DATABASE,
# but with requirements tailored for JudTier
# (we need low latency for relatively large number of tasks,
# so we use MULTISLOT hosts, but they should not be used for ~500G WebTier1)
import logging
import os

from sandbox import sdk2
from datetime import datetime
from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import copy


class JudTierImportShard(sdk2.Task):
    """
    Выкачивает отдельно взятый шард JudTier.
    Вспомогательный таск для JUD_TIER_IMPORT_SHARD_SET
    """

    class Parameters(sdk2.Task.Parameters):
        search_database_rsync_path = sdk2.parameters.String('Shard download link (rbtorrent:NNN)', required=True, description="Full rbtorrent-link to search database")
        database_shard_name = sdk2.parameters.String('Specify a shard name', required=True, description="Download database by given shard name (e.g. primus015-050-<timestamp>)")

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 50 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Task.Context):
        already_downloaded = False
        search_database_resource_id = None

    def on_enqueue(self):
        shard_name = self.Parameters.database_shard_name
        database_resource = resource_types.SEARCH_DATABASE.find(state='READY', attrs={'shard_instance': shard_name}, limit=100, order='+id').first()
        if database_resource is not None:
            self.Context.search_database_resource_id = database_resource.id
            self.Context.already_downloaded = True
            return

        with self.memoize_stage.create_resource:
            db_resource = resource_types.SEARCH_DATABASE(
                self,
                self.Parameters.description,
                self.Parameters.database_shard_name,
                db_type='basesearch',
                tier_name='JudTier',
            )
            self.Context.search_database_resource_id = db_resource.id

    def on_execute(self):
        if self.Context.already_downloaded:
            # do not download anything if it was detected in on_enqueue
            return
        remote_path = self.Parameters.search_database_rsync_path
        database_resource = sdk2.Resource[self.Context.search_database_resource_id]
        database_resource_data = sdk2.ResourceData(database_resource)
        db_path = str(database_resource_data.path)
        if os.path.exists(db_path):
            paths.remove_path(db_path)
        copy.RemoteCopy(remote_path, db_path)(timeout=self.Parameters.kill_timeout)
        eh.ensure(os.path.exists(db_path), 'Basesearch database was not created successfully')
        self._set_attrs(database_resource, db_path)

    def _set_attrs(self, db, path):
        db.shard_instance = self.Parameters.database_shard_name
        db.backup_task = True

        try:
            stamp_tag_list = fu.read_lines(os.path.join(path, 'stamp.TAG'))

            timestamp = int(stamp_tag_list[0].strip())
            timestamp_hr = datetime.utcfromtimestamp(timestamp).strftime('%Y-%m-%d_%H:%M')
            db.database_timestamp = timestamp
            db.shard_timestamp_hr = timestamp_hr
            for line in stamp_tag_list:
                if 'ShardName=' in line:
                    db.shard_name = line.strip().replace('ShardName=', '')
                elif 'robotScatterGroup=' in line:
                    db.robot_scatter_group = line.strip().replace('robotScatterGroup=', '')
                elif 'SearchZone=' in line:
                    db.search_zone = line.strip().replace('SearchZone=', '')

        except Exception as e:
            logging.info("WARNING!\n%s ", e)
