# coding: utf-8

import logging
import os

import sandbox.common.types.client as ctc

import sandbox.sandboxsdk.paths as sdk_paths
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.errors as sdk_errors

from sandbox.projects.common.juggler import base

from sandbox.projects.juggler.resource_types import JUGGLER_CLIENT_BINARY


class OptionalRepoNameParameter(base.RepoNameParameter):
    required = False


class BuildJugglerClientInArcadia(base.BaseBuildJugglerArcadiaTask):
    """
    Builds juggler client in arcadia
    """

    type = 'BUILD_JUGGLER_CLIENT'
    execution_space = 10000
    client_tags = ctc.Tag.LINUX_PRECISE

    TARGET_DIR = "juggler/juggler_client"

    input_parameters = base.BaseBuildJugglerArcadiaTask.input_parameters + [
        OptionalRepoNameParameter,
        base.VersionPostfixParameter
    ]

    def on_execute(self):
        release_dir = self._build_arcadia_target(
            target_platform_flags='--target-platform DEFAULT-LINUX-X86_64 --target-platform-flag MUSL=yes'
        )
        binary_path = os.path.join(release_dir, "juggler-client")
        root_dir = self.abs_path("debroot")

        if not os.path.exists(binary_path):
            raise sdk_errors.SandboxTaskFailureError("Failed to find binary at path %r", binary_path)
        logging.info("Found binary at path %r", binary_path)

        logging.info("Building debian package")

        sdk_paths.make_folder(root_dir, delete_content=True)

        for d in ('debian', 'tools'):
            sdk_paths.copy_path(
                os.path.join(self.get_arcadia_src_dir("arcadia:/arc/trunk/arcadia/" + self.TARGET_DIR), d),
                os.path.join(root_dir, d)
            )

        sdk_paths.copy_path(
            binary_path,
            os.path.join(root_dir, "juggler-client")
        )
        version = self._build_deb(root_dir, upload_to_repo=bool(self.ctx.get(OptionalRepoNameParameter.name)))

        sdk_process.run_process(["strip", "--strip-debug", binary_path], log_prefix='strip')

        self._make_resource(
            path=binary_path,
            version=version,
            description='%s binary' % "juggler-client",
            resource_type=JUGGLER_CLIENT_BINARY
        )
