# coding: utf-8

import re
import os
import platform
import tarfile
import zipfile
import stat
import shutil
import contextlib
import urlparse
import hashlib
import logging
import time

import requests

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import environments as sandbox_environments
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.common.platform import get_platform_alias, get_arch_from_platform
from sandbox.common.types.task import Status
from sandbox.projects import resource_types
from sandbox.projects.common import environments
from sandbox.projects.common.juggler import base
from sandbox.projects.common.juggler import compiler
from sandbox.projects.common.juggler import wheels
from sandbox.projects.juggler.resource_types import JUGGLER_WHEEL

BZ2_EXTENSIONS = ('.tar.bz2', '.tbz')
ZIP_EXTENSIONS = ('.zip', )
GZIP_EXTENSIONS = ('.tar.gz', '.tgz', '.tar')
TAR_EXTENSIONS = ('.tar', )
ARCHIVE_EXTENSIONS = (ZIP_EXTENSIONS + BZ2_EXTENSIONS + TAR_EXTENSIONS + GZIP_EXTENSIONS)

SETUP_TEMPLATE = """
#!/usr/bin/env python
from setuptools import setup, find_packages, Distribution

class BinaryDistribution(Distribution):
    def has_ext_modules(self):
        return True

setup(
    name='{package_name}',
    version='{package_version}',
    packages=find_packages(),
    include_package_data=True,
    distclass=BinaryDistribution
)
""".strip()
MANIFEST_TEMPLATE = """
recursive-include {package_folder} *
""".strip()


def retry_request(url, **kwargs):
    retries = 3
    for idx in xrange(retries):
        try:
            resp = requests.get(url, **kwargs)
        except Exception:
            if idx + 1 == retries:
                raise
            logging.exception("Unable to open %s", url)
            time.sleep(1)
        else:
            return resp
    assert False, "can't reach here"


def download_package(dest, package, version, simple_index):
    package = package.lower().replace("-", "_")
    resp = retry_request("{0}{1}/".format(simple_index, package))
    urls = re.findall(r'href=[\'"]?([^\'" >]+)', resp.text)
    for url in urls:
        name = os.path.basename(url)
        logging.info("Check link %s", name)
        if "#md5=" not in name:
            continue
        name, md5_hash = name.split("#md5=")
        name = name.lower().replace("-", "_")
        if any(name == "{0}_{1}{2}".format(package, version, ext) for ext in ARCHIVE_EXTENSIONS):
            url = urlparse.urljoin(simple_index, url)
            resp = retry_request(url, stream=True)
            hasher = hashlib.md5()
            with open(os.path.join(dest, name), "wb") as fd:
                for chunk in resp.iter_content(chunk_size=4096):
                    hasher.update(chunk)
                    fd.write(chunk)
            if hasher.hexdigest() != md5_hash:
                raise errors.SandboxTaskFailureError("md5 hash mismatch")
            return name
    raise errors.SandboxTaskFailureError("release not found")


def split_leading_dir(path):
    path = path.lstrip('/').lstrip('\\')
    if '/' in path and (('\\' in path and path.find('/') < path.find('\\')) or
                        '\\' not in path):
        return path.split('/', 1)
    elif '\\' in path:
        return path.split('\\', 1)
    else:
        return path, ''


def has_leading_dir(paths):
    """Returns true if all the paths have the same leading path name
    (i.e., everything is in one subdirectory in an archive)"""
    common_prefix = None
    for path in paths:
        prefix, rest = split_leading_dir(path)
        if not prefix:
            return False
        elif common_prefix is None:
            common_prefix = prefix
        elif prefix != common_prefix:
            return False
    return True


def current_umask():
    """Get the current umask which involves having to set it temporarily."""
    mask = os.umask(0)
    os.umask(mask)
    return mask


def unzip_file(filename, location):
    with open(filename, 'rb') as zipfp:
        zip = zipfile.ZipFile(zipfp, allowZip64=True)
        leading = has_leading_dir(zip.namelist())
        for info in zip.infolist():
            name = info.filename
            data = zip.read(name)
            fn = name
            if leading:
                fn = split_leading_dir(name)[1]
            fn = os.path.join(location, fn)
            dir = os.path.dirname(fn)
            if fn.endswith('/') or fn.endswith('\\'):
                # A directory
                paths.make_folder(fn)
            else:
                paths.make_folder(dir)
                with open(fn, 'wb') as fp:
                    fp.write(data)
                mode = info.external_attr >> 16
                # if mode and regular file and any execute permissions for
                # user/group/world?
                if mode and stat.S_ISREG(mode) and mode & 0o111:
                    # make dest file have execute for user/group/world
                    # (chmod +x) no-op on windows per python docs
                    os.chmod(fn, (0o777 - current_umask() | 0o111))


def untar_file(filename, location):
    if filename.lower().endswith(GZIP_EXTENSIONS):
        mode = 'r:gz'
    elif filename.lower().endswith(BZ2_EXTENSIONS):
        mode = 'r:bz2'
    elif filename.lower().endswith(TAR_EXTENSIONS):
        mode = 'r'
    else:
        mode = 'r:*'
    with contextlib.closing(tarfile.open(filename, mode)) as tar:
        # note: python<=2.5 doesn't seem to know about pax headers, filter them
        leading = has_leading_dir([
            member.name for member in tar.getmembers()
            if member.name != 'pax_global_header'
        ])
        for member in tar.getmembers():
            fn = member.name
            if fn == 'pax_global_header':
                continue
            if leading:
                fn = split_leading_dir(fn)[1]
            path = os.path.join(location, fn)
            if member.isdir():
                paths.make_folder(path)
            elif member.issym():
                try:
                    tar._extract_member(member, path)
                except Exception:
                    # Some corrupt tar files seem to produce this
                    # (specifically bad symlinks)
                    continue
            else:
                try:
                    fp = tar.extractfile(member)
                except (KeyError, AttributeError):
                    # Some corrupt tar files seem to produce this
                    # (specifically bad symlinks)
                    continue
                with contextlib.closing(fp):
                    paths.make_folder(os.path.dirname(path))
                    with open(path, 'wb') as destfp:
                        shutil.copyfileobj(fp, destfp)
                # member have any execute permissions for user/group/world?
                if member.mode & 0o111:
                    # make dest file have execute for user/group/world
                    # no-op on windows per python docs
                    os.chmod(path, (0o777 - current_umask() | 0o111))


def unpack_file(filename, location):
    if zipfile.is_zipfile(filename):
        unzip_file(filename, location)
    elif tarfile.is_tarfile(filename):
        untar_file(filename, location)
    else:
        raise errors.SandboxTaskFailureError("Unknown archive type given")


class VersionParameter(parameters.SandboxStringParameter):
    name = 'package_version'
    description = 'Custom version of package'
    default_value = ''
    required = True


class PackageParameter(parameters.SandboxStringParameter):
    name = 'package_name'
    description = 'Name of package to create wheel from'
    default_value = ''
    required = True


class ArcadiaTargetParameter(parameters.SandboxStringParameter):
    name = 'arcadia_target'
    description = 'Project to build'
    default_value = ''


class ArcadiaArtifactParameter(parameters.SandboxStringParameter):
    name = 'arcadia_artifact'
    description = 'Folder to make resource from'
    default_value = ''


class UseSystemPythonParameter(parameters.SandboxBoolParameter):
    name = 'use_system_python'
    description = 'Use system python'
    default_value = False


class BuildJugglerWheel(base.BaseBuildJugglerTask):
    """
    Build wheel for juggler's python.
    """

    type = "BUILD_JUGGLER_WHEEL"
    execution_space = 2000
    svn_patches_dir = "arcadia:/arc/trunk/arcadia/juggler/contrib/wheels"

    environment = base.BaseBuildJugglerTask.environment + [
        environments.SandboxLibSnappyEnvironment(),
        environments.SandboxLibMysqlClientEnvironment(),
        environments.SandboxLibFFIEnvironment()
    ]
    input_parameters = [PackageParameter, VersionParameter,
                        ArcadiaTargetParameter, ArcadiaArtifactParameter,
                        UseSystemPythonParameter]
    pypi_url = "https://pypi.yandex-team.ru/simple/"

    def _build_from_arcadia(self, package_name, package_version):
        build_task_id = self.ctx.get('_build_task_id')
        if build_task_id is None:
            build_task = self.create_subtask(
                task_type='YA_MAKE',
                description='Build {0}=={1}'.format(package_name, package_version),
                input_parameters={
                    'targets': self.ctx.get(ArcadiaTargetParameter.name),
                    'arts': self.ctx.get(ArcadiaArtifactParameter.name),
                    'build_system': 'ya',
                    'result_rt': resource_types.ARCADIA_PROJECT_TGZ.name,
                    'build_type': 'release',
                    'checkout_arcadia_from_url': 'arcadia:/arc/trunk/arcadia@{0}'.format(package_version),
                    'clear_build': True,
                    'check_return_code': True,
                    'use_system_python': self.ctx.get(UseSystemPythonParameter.name, False)
                },
                arch=get_platform_alias(platform.platform())
            )
            self.ctx['_build_task_id'] = build_task.id
            self.wait_tasks(
                tasks=[build_task],
                statuses=tuple(Status.Group.FINISH + Status.Group.BREAK),
                wait_all=True)

        build_task = channel.channel.sandbox.get_task(build_task_id)
        if not build_task.is_done():
            raise errors.SandboxTaskFailureError('ya make failed')
        resources = channel.channel.sandbox.list_resources(
            resource_type=resource_types.ARCADIA_PROJECT_TGZ,
            task_id=build_task_id)
        if not resources or not resources[0].is_ready():
            raise errors.SandboxTaskFailureError('Resource not ready or not found')
        resource_dir = self.sync_resource(resources[0].id)

        package_dir = self.abs_path("package")
        paths.copy_path(resource_dir, package_dir)
        paths.add_write_permissions_for_path(package_dir)
        with open(os.path.join(package_dir, "setup.py"), "w") as fd:
            fd.write(SETUP_TEMPLATE.format(package_name=package_name, package_version=package_version))
        with open(os.path.join(package_dir, "MANIFEST.in"), "w") as fd:
            fd.write(MANIFEST_TEMPLATE.format(package_folder=package_name.split('.', 1)[0]))
        return package_dir

    def _build_from_pypi(self, package_name, package_version):
        patches_dir = self.abs_path("patches")
        Arcadia.export(self.svn_patches_dir, patches_dir)
        if not os.path.exists(patches_dir):
            raise errors.SandboxTaskFailureError("Failed to export wheel patches from svn")

        sdist_dir = self.abs_path("sdist")
        paths.make_folder(sdist_dir)
        sdist_package = download_package(sdist_dir, package_name, package_version, self.pypi_url)

        # unpack downloaded archive
        source_dir = self.abs_path("src")
        paths.make_folder(source_dir)
        unpack_file(os.path.join(sdist_dir, sdist_package), source_dir)

        for patch_file in sorted(paths.list_dir(
                patches_dir, filter="{0}-{1}".format(package_name, package_version),
                abs_path=True, files_only=True)):
            if not patch_file.endswith(".patch"):
                continue
            process.run_process((
                "patch", "-p1", "-i", patch_file
            ), log_prefix="pip_patch_{0}".format(package_name), work_dir=source_dir)

        return source_dir

    def on_execute(self):
        package_name = self.ctx[PackageParameter.name].lower()
        package_version = self.ctx[VersionParameter.name]

        with self._compiler_environ():
            self.venv = sandbox_environments.VirtualEnvironment(self.path("buildenv"), use_system=False)
            if self.ctx.get(UseSystemPythonParameter.name, False):
                self.venv.VIRTUALENV_EXE = 'python2.7'
            with self.venv as venv:
                sandbox_environments.PipEnvironment('pip', version="10.0.1", venv=venv).prepare()

                build_environ = dict(os.environ)
                build_environ.update(
                    (key, compiler.merge_with_environ(key, value))
                    for key, value in venv.get_extra_build_variables().items()
                )
                # reserve more space for rpath
                build_environ["LDFLAGS"] += " -Wl,-rpath,/" + "x" * 128
                # gevent>1.1 don't like -l in CFLAGS, see http://www.gevent.org/whatsnew_1_1.html#library-updates
                if package_name == "gevent":
                    build_environ["CPPFLAGS"] = build_environ.pop("CFLAGS")
                if package_name in ("grpcio", "grpcio-tools"):
                    build_environ["GRPC_PYTHON_LDFLAGS"] = " -lpthread -static-libstdc++"

                pip_path = os.path.join(venv.root_dir, "bin", "pip")
                install_deps = (
                    "setuptools==39.2.0",
                    "setuptools-scm==5.0.2",
                    "six==1.12.0",
                    "wheel==0.29.0",
                )
                if package_name != "cffi":
                    install_deps += ("cffi==1.11.5", )
                process.run_process((
                    venv.executable, pip_path,
                    "install",
                    "--index-url", self.pypi_url,
                    "--upgrade-strategy", "only-if-needed",
                ) + install_deps, log_prefix="pip_deps", environment=build_environ)

                if self.ctx.get(ArcadiaTargetParameter.name):
                    source_dir = self._build_from_arcadia(package_name, package_version)
                else:
                    source_dir = self._build_from_pypi(package_name, package_version)

                wheel_dir = self.abs_path("wheelhouse")
                paths.make_folder(wheel_dir)
                process.run_process((
                    venv.executable, pip_path,
                    "wheel",
                    "--index-url", self.pypi_url,
                    "--verbose", "--no-deps", "--wheel-dir", wheel_dir, source_dir
                ), log_prefix="pip_build", environment=build_environ)
                try:
                    [wheel_package] = os.listdir(wheel_dir)
                except ValueError:
                    raise errors.SandboxTaskFailureError("pip can't build package, see logs")
                abbr_impl = wheels.get_abbr_impl(venv.executable)

        self.create_resource(
            resource_path=os.path.join(wheel_dir, wheel_package),
            resource_type=JUGGLER_WHEEL,
            arch=get_arch_from_platform(platform.platform()),
            attributes={
                "package_name": package_name,
                "version": package_version,
                "platform": platform.platform(),
                "pyimpl": abbr_impl,
                "ttl": 30
            },
            description="wheel for {0} version {1}".format(package_name, package_version))


__Task__ = BuildJugglerWheel
