# coding: utf-8

import re
import os
import tarfile

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import LastReleasedResource, SandboxStringParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.channel import channel
import sandbox.sandboxsdk.svn as sdk_svn
from sandbox.sandboxsdk import task
from sandbox.projects.common.juggler import jclient
from sandbox.projects.juggler.resource_types import JUGGLER_ANSIBLE


class JugglerAnsibleResource(LastReleasedResource):
    name = 'juggler_ansible_resource'
    description = 'Juggler Ansible resource'
    resource_type = JUGGLER_ANSIBLE


class PlaybookNameParameter(SandboxStringParameter):
    name = 'playbook_name'
    description = 'Playbook name'
    default_value = 'main.yml'
    required = True


class RunJugglerSearchPlaybook(task.SandboxTask):
    """
    Apply juggler search playbook.
    """

    type = 'RUN_JUGGLER_SEARCH_PLAYBOOK'

    execution_space = 100

    input_parameters = [
        JugglerAnsibleResource,
        PlaybookNameParameter,
    ]

    client_tags = ctc.Tag.LINUX_PRECISE

    _REPO_URL = "svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/juggler/playbooks/search"

    @staticmethod
    def _extract_tags(data):
        for line in data.splitlines():
            m = re.match(r'^\s+TASK\s+TAGS:\s+\[(.+)\]\s*$', line)
            if m is not None:
                return [x.strip() for x in m.group(1).split(',') if x.strip()]
        raise SandboxTaskFailureError('Unable to find tags to run')

    def on_execute(self):
        juggler_ansible_resource = self.ctx.get('juggler_ansible_resource')
        if not juggler_ansible_resource or (juggler_ansible_resource == '0'):
            juggler_ansible_resource = channel.sandbox.list_releases(
                resource_type=JUGGLER_ANSIBLE, limit=1
            )[0].resources[0].id
            self.ctx['juggler_ansible_resource'] = juggler_ansible_resource

        resource_path = self.sync_resource(juggler_ansible_resource)
        venv_path = self.path('juggler_ansible')
        tarfile.open(resource_path).extractall(venv_path)

        python_path = os.path.join(venv_path, 'bin', 'python')
        ansible_playbook_path = os.path.join(venv_path, 'bin', 'ansible-playbook')
        dynamic_playbook_path = os.path.join(venv_path, 'bin', 'dynamic-juggler-playbook')

        checkout_path = self.path("root")
        sdk_svn.Svn.checkout(url=self._REPO_URL, path=checkout_path)

        inventory_path = os.path.join(checkout_path, 'hosts')
        with open(inventory_path, 'w'):
            pass

        playbook_argv = ['--inventory-file={0}'.format(inventory_path), self.ctx['playbook_name']]
        environment = os.environ.copy()
        environment['NANNY_OAUTH'] = self.get_vault_data('JUGGLER', 'robot-juggling-nanny-oauth')
        environment['JUGGLER_OAUTH_TOKEN'] = self.get_vault_data('JUGGLER', 'robot-juggling-juggler-oauth')

        proc = run_process(
            [python_path, ansible_playbook_path, '--list-tags'] + playbook_argv,
            outs_to_pipe=True, work_dir=checkout_path)
        known_tags = self._extract_tags(proc.communicate()[0])
        processed_tags = 0

        if 'static' in known_tags:
            run_process(
                [python_path, ansible_playbook_path, '--tags', 'static'] + playbook_argv,
                log_prefix='static_playbook', work_dir=checkout_path,
                environment=environment)
            processed_tags += 1

        if 'dynamic' in known_tags:
            run_process(
                [python_path, dynamic_playbook_path] + playbook_argv,
                log_prefix='dynamic_playbook', work_dir=checkout_path,
                environment=environment)
            processed_tags += 1

        service_name = 'ansible_{0}'.format(self.ctx['playbook_name'].replace('.', '_'))
        if processed_tags:
            jclient.send_events_to_juggler('juggler.sandbox', service_name, 'OK', 'Passed')
        else:
            jclient.send_events_to_juggler('juggler.sandbox', service_name, 'WARN', 'No tags was processed')
