import json
import logging
import os
from os.path import join as pj

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.jupiter.CompareIndex import html
from sandbox.projects.jupiter.CompareIndex import tool
from sandbox.projects.jupiter.CompareIndex.result import JupiterCompareIndexResult


class JupiterCompareIndex(sdk2.Task):
    """
    Compare indexes of mercury-SaaS, callisto-mercury and jupiter.
    """

    JUPITER_PATH = "robot/jupiter/test/integration-diff/test-results/pytest/testing_out_stuff/shards"
    MERCURY_INDEXER_PATH = "robot/mercury/test/comparsion_indexer_test/test-results/pytest/testing_out_stuff/index"
    CALLISTO_MERCURY_PATH = "robot/mercury/test/comparsion_callisto_test/test-results/pytest/testing_out_stuff/shards"

    class Parameters(sdk2.Task.Parameters):
        jupiter_output = sdk2.parameters.Resource(
            "JUPITER_INTEGRATION_DIFF_TEST output ARCADIA_PROJECT resource",
            resource_type=resource_types.ARCADIA_PROJECT,
            required=True)

        mercury_indexer_output = sdk2.parameters.Resource(
            "COMPARSION_MERCURY_INDEXER_TEST output ARCADIA_PROJECT resource",
            resource_type=resource_types.ARCADIA_PROJECT,
            required=True)

        callisto_mercury_output = sdk2.parameters.Resource(
            "COMPARSION_CALLISTO_MERCURY_TEST output ARCADIA_PROJECT resource",
            resource_type=resource_types.ARCADIA_PROJECT,
            required=True)

        compare_tool = sdk2.parameters.Resource(
            "Compare tool resource (get the latest released one by default)",
            resource_type=tool.JupiterCompareIndexTool)

    def on_execute(self):
        self._prepare()
        self._compare_file_list()
        self._compare_index()

        # Save results
        self.output_resource = JupiterCompareIndexResult(self, "Compare index result", "result")
        output = sdk2.ResourceData(self.output_resource)
        output.path.mkdir(0o755, parents=True, exist_ok=True)
        output.path.joinpath(JupiterCompareIndexResult.FileName.FILE_LIST_JSON).write_bytes(self.file_list_result_json)

        file_list_result_html = html.get_file_list_result(json.loads(self.file_list_result_json))
        self.Context.file_list_result_html = file_list_result_html
        output.path.joinpath(JupiterCompareIndexResult.FileName.FILE_LIST_HTML).write_bytes(file_list_result_html)

        for name, result_json in self.compare_index_result_json.iteritems():
            subdir_path = output.path.joinpath(name)
            subdir_path.mkdir(0o755, parents=True, exist_ok=True)
            subdir_path.joinpath(JupiterCompareIndexResult.FileName.COMPARE_INDEX_JSON).write_bytes(result_json)
            self._write_files(subdir_path, name, result_json)

        output.ready()

        self._save_brief()

        self.Context.save()

    def _save_brief(self):
        links = []
        link_base = self.server.resource[self.output_resource.id][:]["http"]["proxy"]

        for name in sorted(self.compare_index_result_json):
            display_name = name.replace("_", " ")
            links.append([display_name, link_base + "/" + name])

        self.Context.compare_index_brief_html = html.get_brief_html(links)

    @staticmethod
    def _write_files(path, name, result_json):
        result = json.loads(result_json)
        for mode, diff in result["diff"].iteritems():
            if diff:
                filename = JupiterCompareIndexResult.FileName.get_diff_name(mode)
                path.joinpath(filename).write_bytes(diff.encode('utf-8'))

        lost_docs_html = html.get_lost_docs_result(result["lost_docs"], *name.split("_vs_"))
        path.joinpath(JupiterCompareIndexResult.FileName.COMPARE_INDEX_LOST_DOCS_HTML).write_bytes(lost_docs_html)

    @staticmethod
    def _get_shard_list(path):
        return [pj(path, shard_name) for shard_name in os.listdir(path)]

    def _prepare(self):
        self.jupiter_shard_paths = self._get_shard_list(pj(str(sdk2.ResourceData(self.Parameters.jupiter_output).path), self.JUPITER_PATH))
        self.mercury_indexer_shard_paths = [pj(str(sdk2.ResourceData(self.Parameters.mercury_indexer_output).path), self.MERCURY_INDEXER_PATH)]  # one shard only
        self.callisto_mercury_shard_paths = self._get_shard_list(pj(str(sdk2.ResourceData(self.Parameters.callisto_mercury_output).path), self.CALLISTO_MERCURY_PATH))

        if self.Parameters.compare_tool:
            self.compare_tool_resource = self.Parameters.compare_tool
        else:
            self.compare_tool_resource = tool.get_last_released()

        logging.info("Compare tool resource id: {}".format(self.compare_tool_resource.id))

        self.compare_tool_binary = str(sdk2.ResourceData(self.compare_tool_resource).path)

    def _compare_file_list(self):
        cmd_args = [
            self.compare_tool_binary,
            "CompareFileList",
            "--basic-path", self.jupiter_shard_paths[0],
            "--check-path", self.mercury_indexer_shard_paths[0],
            "--check-path", self.callisto_mercury_shard_paths[0],
        ]

        with sdk2.helpers.ProcessLog(self, logger="compare_tool_CompareFileList") as pl:
            self.file_list_result_json = sp.check_output(cmd_args, stderr=pl.stdout)

    def _compare_shards(self, first_shard_paths, second_shard_paths, name):
        cmd_args = [
            self.compare_tool_binary,
            "CompareIndex"
        ]
        for shard_path in first_shard_paths:
            cmd_args += ["--first-index-shard-path", shard_path]
        for shard_path in second_shard_paths:
            cmd_args += ["--second-index-shard-path", shard_path]

        with sdk2.helpers.ProcessLog(self, logger="compare_tool_CompareIndex_{}".format(name)) as pl:
            self.compare_index_result_json[name] = sp.check_output(cmd_args, stderr=pl.stdout)

    def _compare_index(self):
        self.compare_index_result_json = {}
        self._compare_shards(self.callisto_mercury_shard_paths, self.mercury_indexer_shard_paths, "Callisto-Mercury_vs_Mercury-SaaS")
        # should add comparison with Jupiter too

    @sdk2.report(title="Compare file list result")
    def report_file_list(self):
        return self.Context.file_list_result_html or "No result"

    @sdk2.report(title="Compare index diff")
    def report_brief(self):
        return self.Context.compare_index_brief_html or "No result"
