import json
import logging
import os
from os.path import join as pj

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.jupiter.CompareIndex import html
from sandbox.projects.jupiter.CompareIndex import tool
from sandbox.projects.jupiter.CompareIndex.result import JupiterCompareIndexResult
from sandbox.projects.jupiter.CompareIndexDiff.result import JupiterCompareIndexDiffBriefResult, JupiterCompareIndexDiffDetailedResult


class JupiterCompareIndexDiff(sdk2.Task):
    """
    Make diff of compare index results
    """

    class Parameters(sdk2.Task.Parameters):
        first_compare_index_result = sdk2.parameters.Resource(
            "First JUPITER_COMPARE_INDEX output resource",
            resource_type=JupiterCompareIndexResult,
            required=True)

        second_compare_index_result = sdk2.parameters.Resource(
            "Second JUPITER_COMPARE_INDEX output resource",
            resource_type=JupiterCompareIndexResult,
            required=True)

        compare_tool = sdk2.parameters.Resource(
            "Compare tool resource (get the latest released one by default)",
            resource_type=tool.JupiterCompareIndexTool)

    def on_execute(self):
        self._prepare()
        self._get_file_list_diff()
        self._get_index_diff()

        self.output_detailed_resource = JupiterCompareIndexDiffDetailedResult(self, "Detailed diff changes", "result")
        output_detailed = sdk2.ResourceData(self.output_detailed_resource)
        output_detailed.path.mkdir(0o755, parents=True, exist_ok=True)

        for name, result_json in self.index_diff_jsons.iteritems():
            subdir_path = output_detailed.path.joinpath(name)
            subdir_path.mkdir(0o755, parents=True, exist_ok=True)
            self._write_files(subdir_path, name, json.loads(result_json))

        self._save_info(output_detailed)

        output_detailed.ready()

        self._save_brief()

        self.Context.save()

    def _save_info(self, output_detailed):
        info = "Compare resources: {} vs {}".format(
            self.Parameters.first_compare_index_result.id,
            self.Parameters.second_compare_index_result.id,
        )
        output_detailed.path.joinpath(JupiterCompareIndexDiffDetailedResult.FileName.INFO_TXT).write_bytes(info)

    def _save_brief(self):
        links = []
        link_base = self.server.resource[self.output_detailed_resource.id][:]["http"]["proxy"]

        for name in sorted(self.index_diff_jsons):
            display_name = name.replace("_", " ")
            links.append([display_name, link_base + "/" + name])

        diff_html = html.get_diff_result(
            json.loads(self.file_list_diff_json) if self.file_list_diff_json else None,
            links,
        )

        if self.file_list_diff_json or self.index_diff_jsons:
            self.Context.has_diff = True
        else:
            self.Context.has_diff = False

        output_brief = sdk2.ResourceData(JupiterCompareIndexDiffBriefResult(self, "Diff summary", "diff.html"))
        output_brief.path.write_bytes(diff_html)
        output_brief.ready()

        self.Context.diff_html = diff_html

    @staticmethod
    def _write_files(path, name, result):
        lost_docs = result.get("lost_docs", None)
        if lost_docs:
            lost_docs_html = html.get_lost_docs_diff_result(lost_docs, *name.split("_vs_"))
            path.joinpath(JupiterCompareIndexDiffDetailedResult.FileName.LOST_DOCS_HTML).write_bytes(lost_docs_html)

        mode_diffs = result.get("diff", {})
        for mode, diff in mode_diffs.iteritems():
            if diff:
                filename = JupiterCompareIndexDiffDetailedResult.FileName.get_diff_name(mode)
                path.joinpath(filename).write_bytes(diff.encode('utf-8'))

    def _prepare(self):
        self.result_paths = [
            str(sdk2.ResourceData(self.Parameters.first_compare_index_result).path),
            str(sdk2.ResourceData(self.Parameters.second_compare_index_result).path),
        ]

        if self.Parameters.compare_tool:
            self.compare_tool_resource = self.Parameters.compare_tool
        else:
            self.compare_tool_resource = tool.get_last_released()

        logging.info("Compare tool resource id: {}".format(self.compare_tool_resource.id))

        self.compare_tool_binary = str(sdk2.ResourceData(self.compare_tool_resource).path)

    def _get_file_list_diff(self):
        cmd_args = [
            self.compare_tool_binary,
            "GetFileListDiff",
            "--first-result-path", pj(self.result_paths[0], JupiterCompareIndexResult.FileName.FILE_LIST_JSON),
            "--second-result-path", pj(self.result_paths[1], JupiterCompareIndexResult.FileName.FILE_LIST_JSON),
        ]

        with sdk2.helpers.ProcessLog(self, logger="compare_tool_GetFileListDiff") as pl:
            self.file_list_diff_json = sp.check_output(cmd_args, stderr=pl.stdout)

    def _get_result_diff_names(self):
        dir_names = []  # list of sets
        for result_path in self.result_paths:
            names = os.listdir(result_path)
            dir_names.append(set(filter(lambda name: os.path.isdir(pj(result_path, name)), names)))

        return list(set.intersection(*dir_names))

    def _get_index_diff(self):
        self.index_diff_jsons = {}
        dir_names = self._get_result_diff_names()
        logging.info("Check changes for " + ", ".join(dir_names))

        for dir_name in dir_names:
            cmd_args = [
                self.compare_tool_binary,
                "GetIndexDiff",
                "--first-result-path", pj(self.result_paths[0], dir_name, JupiterCompareIndexResult.FileName.COMPARE_INDEX_JSON),
                "--second-result-path", pj(self.result_paths[1], dir_name, JupiterCompareIndexResult.FileName.COMPARE_INDEX_JSON),
            ]

            with sdk2.helpers.ProcessLog(self, logger="compare_tool_GetIndexDiff_{}".format(dir_name)) as pl:
                index_diff_json = sp.check_output(cmd_args, stderr=pl.stdout)
                if index_diff_json:  # something changed
                    self.index_diff_jsons[dir_name] = index_diff_json

    @sdk2.report(title="Diff")
    def report_diff(self):
        return self.Context.diff_html or "No result"
