import logging
import subprocess
from os.path import join as pj
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.common.nanny import nanny


class GeminiData(sdk2.Resource):
    releasable = True
    any_arch = True
    executable = False
    auto_backup = True
    calc_md5 = True
    share = True
    ttl = 100
    releasers = ['bvshuryg']


def download_file(yt_client, path, local_path):
    from yt.wrapper.common import chunk_iter_stream, MB
    logging.info("Downloading yt file %r to %r", path, local_path)
    with open(local_path, "wb") as fout:
        for chunk in chunk_iter_stream(yt_client.read_file(path), 16 * MB):
            fout.write(chunk)


class JupiterCreateGeminiData(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Task downloads gemini data files from yt, tar/zips them and uploads gemini.data.tar.gz to sandbox.
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 4 * 1024  # 4GB

        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]

    class Parameters(sdk2.Task.Parameters):
        mr_prefix = sdk2.parameters.String("YT prefix", default="//home/jupiter")
        mr_server = sdk2.parameters.String("YT proxy", default="arnold.yt.yandex.net")
        state = sdk2.parameters.String("Gemini state", default="Last")
        output = sdk2.parameters.String("Name of resulting file", default="gemini.data.tar.gz")
        files = sdk2.parameters.List("List of input file names", default=["mirrors.trie", "mirrors_host_map", "filter.rfl.mapped", "filter.robots.rfl.mapped"])
        yt_token_name = sdk2.parameters.String(
                    "Name of the sandbox vault record storing yt token", default="jupiter_yt_token")
        yt_token_owner = sdk2.parameters.String(
                    "Owner of the sandbox vault record storing yt token", default="JUPITER")

    def on_execute(self):
        from yt.wrapper import YtClient
        yt_token = sdk2.Vault.data(self.Parameters.yt_token_owner, self.Parameters.yt_token_name)
        yt_client = YtClient(proxy=self.Parameters.mr_server, token=yt_token)

        state = self.Parameters.state

        if state == "Last":
            state = yt_client.get(pj(self.Parameters.mr_prefix, "@jupiter_meta", "yandex_current_state"))

        for infile in self.Parameters.files:
            download_file(yt_client,
                pj(self.Parameters.mr_prefix, "gemini", state, infile),
                infile
            )

        logging.info("Packing files to %s.", self.Parameters.output)
        subprocess.check_call(["tar", "zcvf", self.Parameters.output] + self.Parameters.files)

        data = sdk2.ResourceData(GeminiData(self, self.Parameters.output, self.Parameters.output))
        data.ready()
