# -*- coding: utf-8 -*-

import logging
import os
from os.path import join as pj
import subprocess
import tempfile
import json

from sandbox import sdk2
from sandbox import common
from sandbox.common.types.task import Status, ReleaseStatus
from sandbox.common.types.resource import State
from sandbox.projects.resource_types import GEMINICL
from sandbox.sandboxsdk.copy import RemoteCopy


class JupiterNavsourceData(sdk2.Resource):
    any_arch = True
    executable = False
    releasable = True
    release_subscribers = ["chelentano"]
    auto_backup = True
    calc_md5 = True
    share = True


class JupiterCanonizedNavsourceData(sdk2.Resource):
    any_arch = True
    executable = False
    releasable = True
    release_subscribers = ["chelentano"]
    auto_backup = True
    calc_md5 = True
    share = True


class LastGeminiCL(sdk2.parameters.Resource):
    resource_type = GEMINICL

    @common.utils.classproperty
    def default_value(cls):
        return sdk2.Resource.find(type=cls.resource_type, state=State.READY).first().id


class JupiterReleaseNavsource(sdk2.Task):
    """
    Release of NavSource data
    """

    GEMINICL_USER = "robot-jupiter-spot"
    GEMINICL_MAX_RPS = "100"  # from arcadia/robot/deprecated/gemini/configs/squota.gemini.xml

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1024  # Mb
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        geminicl_resource_id = LastGeminiCL('geminicl tool', required=True)

    def on_execute(self):
        if self.parent:
            self._build_navsource()
        else:
            build_task = self.find().first() or JupiterReleaseNavsource(
                self,
                description="Child task of {} to build navsource data".format(self.id),
                geminicl_resource_id=self.Parameters.geminicl_resource_id,
                create_sub_task=False
            ).enqueue()

            expected_statuses = Status.Group.FINISH + Status.Group.BREAK
            if build_task.status not in expected_statuses:
                raise sdk2.WaitTask([build_task], expected_statuses)

            if build_task.status not in [Status.SUCCESS, Status.RELEASED]:
                raise common.errors.TaskFailure("Build navsource task {} has been failed with status {}".format(build_task.id, build_task.status))

            if build_task.status == Status.SUCCESS:
                self.server.release({
                    'task_id': build_task.id,
                    'type': ReleaseStatus.STABLE,
                    'subject': 'Release of navsource data',
                })
                raise sdk2.WaitTask([build_task], expected_statuses)

    def _copy_export(self, export, local_dir, markers_txt):
        r = RemoteCopy("http://navig-backend.n.yandex-team.ru/export/" + export, pj(local_dir, markers_txt))
        logging.error(r)
        logging.info("Using %r to fetch remote data", r)
        r(timeout=RemoteCopy.DEFAULT_TIMEOUT)

    def _handle_export(self, export, local_dir, canonized_local_dir, markers_txt):
        self._copy_export(export, local_dir, markers_txt)

        urls_file = tempfile.NamedTemporaryFile()
        canonized_urls_file = tempfile.NamedTemporaryFile()

        canonize_urls = {}
        with open(pj(local_dir, markers_txt)) as fin:
            for rec in fin:
                descr, url, v1, v2, v3 = rec.split("\t")
                canonize_urls[url] = url

        for url in canonize_urls:
            urls_file.write(url + "\n")
        urls_file.flush()

        geminicl_cmd = [str(sdk2.ResourceData(self.Parameters.geminicl_resource_id).path),
            "-t", "mirror", "--quota", self.GEMINICL_MAX_RPS, "-f", urls_file.name, "--format", "json", "-u", self.GEMINICL_USER]
        logging.info("Running %s", " ".join(geminicl_cmd))

        subprocess.check_call(geminicl_cmd, stdout=canonized_urls_file)
        canonized_urls_file.flush()

        with open(canonized_urls_file.name) as fin:
            for rec in fin:
                response = json.loads(rec)["Response"]
                original_url = response["OriginalUrl"]
                if "Error" in response:
                    logging.error(response)
                    if int(response["Error"]) > 0:
                        raise common.errors.TaskFailure("{} canonize failure. returns {}".format(original_url, response.get("ErrorMessage") or response.get("Error")))
                else:
                    legacy_canonized_url = response.get("MainMirror", None)  # old geminicl output format
                    canonized_url = response.get("CanonizedUrl", None)
                    if legacy_canonized_url is None and canonized_url is None:
                        logging.error(response)
                        raise common.errors.TaskFailure("{} canonize failure. Missing canonization result".format(original_url))
                    elif (legacy_canonized_url is not None) and (canonized_url is not None) and (legacy_canonized_url != canonized_url):
                        logging.error(response)
                        raise common.errors.TaskFailure("{} canonize failure. Ambiguous canonization result: {} and {}".format(original_url, legacy_canonized_url, canonized_url))
                    canonize_urls[original_url] = canonized_url or legacy_canonized_url

        with open(pj(canonized_local_dir, markers_txt), "w") as fout:
            with open(pj(local_dir, markers_txt)) as fin:
                for rec in fin:
                    descr, url, v1, v2, v3 = rec.split("\t")
                    navstr = "\t".join([descr.decode("utf-8"), canonize_urls[url].decode("utf-8"), v1, v2, v3])
                    fout.write(navstr.encode("utf-8"))

    def _build_navsource(self):
        navsource_dir = str(self.path(JupiterNavsourceData.name))
        os.makedirs(navsource_dir)

        canonized_navsource_dir = str(self.path(JupiterCanonizedNavsourceData.name))
        os.makedirs(canonized_navsource_dir)

        self._copy_export('nomarkers', navsource_dir, 'nomarkers.utf8.txt')
        self._handle_export('markers', navsource_dir, canonized_navsource_dir, 'markers.utf8.txt')
        self._handle_export('bro', navsource_dir, canonized_navsource_dir, 'markers_to_browser.txt')

        nav_resource = JupiterNavsourceData(
            self,
            '<Navigation source> original urls data',
            navsource_dir
        )
        nav_resource_data = sdk2.ResourceData(nav_resource)

        canonized_nav_resource = JupiterCanonizedNavsourceData(
            self,
            '<Navigation source> canonized urls data',
            canonized_navsource_dir
        )
        canonized_nav_resource_data = sdk2.ResourceData(canonized_nav_resource)

        canonized_nav_resource_data.ready()
        nav_resource_data.ready()
