# -*- coding: utf-8 -*-

import os
from os.path import join as pj
import json
import subprocess as sp
import logging
import time

import sandbox.common.types.resource as ctr
from sandbox import sdk2


class JupiterUploadFilesToYt(sdk2.Task):
    '''
        Upload external files to yt
    '''

    meta_file_name = 'meta.json'

    class Requirements(sdk2.Task.Requirements):
        disk_space = 150 * 1024  # Mb

    class Parameters(sdk2.Task.Parameters):
        resource_name_to_upload = sdk2.parameters.String(
            'Arbitrary resource name to upload', required=True)

        upload_name = sdk2.parameters.String(
            'Upload name', required=True)

        delivery_resource_id = sdk2.parameters.LastReleasedResource(
            'Delivery package',
            resource_type=sdk2.Resource['DELIVERY_BINARIES'],
            state=(ctr.State.READY,),
            required=True)

        mr_server = sdk2.parameters.String(
            'Yt proxy', default='banach.yt.yandex.net', required=True)

        mr_prefix = sdk2.parameters.String(
            'Directory', default='//home/jupiter', required=True)

        use_filename_as_table_name = sdk2.parameters.Bool(
            'Use filename as table name for sharded uploads.')

        postfixes_to_strip = sdk2.parameters.String(
            'Space separated postfix list to cut from filenames.', default='')

        sample_urls_table = sdk2.parameters.String(
            'Path to table with sample urls.')

        no_sandbox_upload = sdk2.parameters.Bool(
            'A resource was created without sandbox_upload delivery target.', required=True)

    def unpack_delivery(self):
        delivery_package_path = str(sdk2.ResourceData(self.Parameters.delivery_resource_id).path)

        if delivery_package_path.endswith('.tar'):
            cmd_list = ['tar', '-xf', delivery_package_path]
            with sdk2.helpers.ProcessLog(self, logger='tar') as pl:
                logging.info('Running %s', ' '.join(cmd_list))
                sp.check_call(
                    cmd_list,
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                )

            return os.path.abspath('bin/delivery')
        else:
            return pj(delivery_package_path, 'delivery')

    def on_execute(self):
        jupiter_token = sdk2.Vault.data('JUPITER', 'jupiter_yt_token')
        res_to_upload_inst = sdk2.Resource.find(
            type=self.Parameters.resource_name_to_upload, state=ctr.State.READY).first()
        res_to_upload = str(sdk2.ResourceData(res_to_upload_inst).path)

        res_timestamp = int(time.mktime(res_to_upload_inst.created.timetuple()))
        meta_file_name = pj(res_to_upload, self.meta_file_name)

        meta = {} if self.Parameters.no_sandbox_upload else json.load(open(meta_file_name))

        delivery_path = self.unpack_delivery()

        filename_list = os.listdir(res_to_upload)
        for filename in filename_list:
            if filename == self.meta_file_name:
                continue

            if filename not in meta:
                meta[filename] = res_timestamp

            upload_timestamp = meta[filename]['timestamp'] if isinstance(meta[filename], dict) else meta[filename]
            cmd_list = [
                delivery_path,
                self.Parameters.upload_name,
                'Upload',
                '--server-name', self.Parameters.mr_server,
                '--delivery-prefix', self.Parameters.mr_prefix,
                '--local-file-path', pj(res_to_upload, filename),
                '--upload-timestamp', str(upload_timestamp),
            ]

            if self.Parameters.sample_urls_table:
                cmd_list.append('--sample-keys-table')
                cmd_list.append(self.Parameters.sample_urls_table)

            if self.Parameters.use_filename_as_table_name:
                postfix_list = self.Parameters.postfixes_to_strip.split()
                for postfix in postfix_list:
                    if filename.endswith(postfix):
                        filename = filename[:-len(postfix)]
                cmd_list.append('--table-name')
                cmd_list.append(filename)

            with sdk2.helpers.ProcessLog(self, logger='delivery') as pl:
                logging.info('Running %s', ' '.join(cmd_list))
                sp.check_call(
                    cmd_list,
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                    env={'YT_TOKEN': jupiter_token}
                )

        return 0
