# -*- coding: utf-8 -*-

from __future__ import absolute_import, division, print_function

import logging
from base64 import b64decode

from sandbox import sdk2
from sandbox.projects.jupyter_cloud.common.task_mixin import JupyterCloud3Task

CONCURRENCY = 64
HOUR = 60 * 60
STATE_APPLY_TIMEOUT = 3 * HOUR
TASK_TIMEOUT = STATE_APPLY_TIMEOUT + HOUR


class JupyterCloudDismissUsers(JupyterCloud3Task):
    class Requirements(sdk2.Task.Requirements):
        """Multislot task requirements:
        - cores <= 16
        - ram <= 64GiB
        - no root
        - empty Caches
        """

        cores = 1
        disk_space = 2048
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(JupyterCloud3Task.Parameters):
        kill_timeout = TASK_TIMEOUT

        with sdk2.parameters.RadioGroup(
            'JupyterCloud environment', required=True, hint=True
        ) as environment:
            environment.values['production'] = environment.Value(default=True)
            environment.values['testing'] = environment.Value()

        dry_run = sdk2.parameters.Bool('Dry run', default=True)

    def on_prepare(self):
        shared_secrets = self.Parameters.yav_shared_secret.data()

        self.secrets = {}
        self.secrets['yav_oauth'] = shared_secrets['yav-oauth-token']
        self.secrets['qyp_oauth'] = shared_secrets['qyp_oauth_token']
        self.secrets['minion_ssh_key'] = b64decode(
            shared_secrets['id_rsa']
        )  # files are base-64 encoded

        if self.Parameters.environment == 'production':
            env_secrets = self.Parameters.yav_prod_secret.data()
        else:
            env_secrets = self.Parameters.yav_test_secret.data()

        self.secrets['salt_secret'] = env_secrets['salt_secret']

    def on_execute(self):
        from sandbox.projects.jupyter_cloud.dismiss_users.lib import DismissJobLib

        self.logger = logging.getLogger('JCDismissUsers')

        job = DismissJobLib(job=self)
        job.execute()
