import asyncio
import functools
import logging

import attr
from traitlets.config import Application, Config

from sandbox import sdk2
from sandbox.sdk2.helpers import ProgressMeter

ENVIRONMENTS = {
    'testing': {
        'vm_name_prefix': 'testing-jupyter-cloud-',
        'vm_short_name_prefix': 'tjc-',
        'oauth_id': '82ede8f30a9347379538bc0877730928',
    },
    'production': {
        'vm_name_prefix': 'jupyter-cloud-',
        'vm_short_name_prefix': 'jc-',
        'oauth_id': '0bdc337f290748b982b1e8bc0c345cca',
    },
}

STEPS = 0
GLOBAL_METER = None


def global_progress(title, repetitions=1):
    """Add to functions to track global progress over Sandbox Job
    ```
        class SandboxJob:
            @global_progress('Step 1')
            def f():
                ...

            @global_progress('Step 2')
            def g():
                ...
    """

    def decorator(func):
        global STEPS
        STEPS += repetitions

        @functools.wraps(func)
        def wrapper_decorator(*args, **kwargs):
            global GLOBAL_METER, STEPS
            GLOBAL_METER.message = title
            logging.info('Sandbox Job step %d/%d: %s', GLOBAL_METER.value + 1, STEPS, title)

            value = func(*args, **kwargs)

            GLOBAL_METER.add(repetitions)
            return value

        return wrapper_decorator

    return decorator


@attr.define()
class DismissJobLib:
    job: sdk2.Task = None

    def __getattr__(self, item):
        if self.job is None:
            raise AttributeError
        return getattr(self.job, item)

    app: Application = None
    c: Config = None
    log: logging.Logger = logging.getLogger('JCDismissUsers')

    def execute(self):
        asyncio.run(self.execute_async())

    async def execute_async(self):
        with ProgressMeter('Total task progress', minval=0, maxval=STEPS) as total:
            global GLOBAL_METER
            GLOBAL_METER = total

            self.setup()

    @global_progress('Setting up the environment')
    def setup(self):
        self.log.info('Hello, world!')
