# -*- coding: utf-8 -*-

from __future__ import absolute_import, print_function, division

import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.projects.jupyter_cloud.backup import JupyterCloudBackup
from sandbox.projects.jupyter_cloud.task_mixin import JupyterCloudTaskMixin

logger = logging.getLogger(__name__)

BACKUP_SEMAPHORE_NAME = 'jupyter-cloud-backup'

HOUR = 60 * 60
SUBTASK_TIMEOUT = 12 * HOUR
WAIT_TIMEOUT = SUBTASK_TIMEOUT + HOUR
TASK_TIMEOUT = WAIT_TIMEOUT + HOUR
SUBTASK_TERMINAL_STATUSES = (
    ctt.Status.SUCCESS,
    ctt.Status.FAILURE,
    ctt.Status.EXCEPTION,
    ctt.Status.TIMEOUT,
    ctt.Status.STOPPED,
    ctt.Status.EXPIRED,
    ctt.Status.NO_RES,
    ctt.Status.DELETED,
)


class JupyterCloudMassBackup(JupyterCloudTaskMixin, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        cores = 2
        disk_space = 1 * 1024
        ram = 2 * 1024
        tasks_resource = None

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(JupyterCloudTaskMixin.Parameters):
        kill_timeout = TASK_TIMEOUT

        with sdk2.parameters.Group('Backup/Restore options') as backup_options:
            backup_ttl = sdk2.parameters.Integer('ttl for backup resource', default=183)

            users_to_backup = sdk2.parameters.List(
                'Users to backup',
                value_type=sdk2.parameters.Staff,
                default=[],
                description='count as all available users if not set'
            )

    def on_execute(self):
        from jupytercloud.tools.lib import environment, utils

        yav_token = self.Parameters.yav_oauth_token.data()
        with environment.environment('production', yav_token):
            utils.setup_sentry(__name__)

            with self.memoize_stage.stage_one:
                self.get_user_hosts()

                self.spawn_backup_taks()

                raise sdk2.WaitTask(
                    self.Context.subtask_ids,
                    SUBTASK_TERMINAL_STATUSES,
                    wait_all=True,
                    timeout=WAIT_TIMEOUT,
                )

            with self.memoize_stage.stage_two:
                self.check_subtasks_status()

            if self.Context.failed_users:
                raise TaskFailure(
                    'Failed to backup users: {}'
                    .format(', '.join(self.Context.failed_users))
                )

    def get_user_hosts(self):
        from jupytercloud.tools.lib import JupyterCloud

        jupyter_cloud = JupyterCloud()

        users = [
            user for user in self.Parameters.users_to_backup or []
            if user
        ]

        result = {}

        vms_info = jupyter_cloud.get_qyp_vms(users)

        with sdk2.helpers.ProgressMeter(
            'Fetching vm statuses...', maxval=len(vms_info)
        ) as progress:
            for user, vm_info in vms_info.iteritems():
                status = jupyter_cloud.get_vm_status(
                    cluster=vm_info['cluster'],
                    id=vm_info['id']
                )
                host = vm_info['host']

                if status == 'RUNNING':
                    result[user] = host
                    progress.add(1)
                else:
                    logger.info('vm %s have status %s, skipping', host, status)

        self.Context.user_hosts = result
        self.Context.save()

    def spawn_backup_taks(self):
        subtask_ids = []
        with sdk2.helpers.ProgressMeter(
            'Spawning backup subtasks...', maxval=len(self.Context.user_hosts)
        ) as progress:
            for user, host in self.Context.user_hosts.iteritems():
                task = JupyterCloudBackup(
                    self,
                    kill_timeout=SUBTASK_TIMEOUT,
                    use_latest_sandbox_binary=True,
                    yav_secret_id=self.Parameters.yav_secret_id,
                    yav_oauth_token=self.Parameters.yav_oauth_token,
                    username=user,
                    do_backup=True,
                    old_host=host,
                    backup_ttl=self.Parameters.backup_ttl,
                    description="backup for user {} from host {}".format(user, host),
                    semaphores=ctt.Semaphores(
                        acquires=[
                            ctt.Semaphores.Acquire(name=BACKUP_SEMAPHORE_NAME)
                        ]
                    )
                )

                task.enqueue()
                subtask_ids.append(task.id)
                progress.add(1)

        self.Context.subtask_ids = subtask_ids
        self.Context.save()

    def check_subtasks_status(self):
        failed_users = {}
        with sdk2.helpers.ProgressMeter(
            'Checking backup statuses...', maxval=len(self.Context.subtask_ids)
        ) as progress:
            for subtask_id in self.Context.subtask_ids:
                subtask = sdk2.Task[subtask_id]
                username = subtask.Parameters.username
                if subtask.status != ctt.Status.SUCCESS:
                    failed_users[username] = subtask_id

                progress.add(1)

        self.Context.failed_users = failed_users
        self.Context.save()
