# -*- coding: utf-8 -*-

from __future__ import absolute_import, print_function, division

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import resource as ctr

JUPYTERCLOUD_SECRETS_ID = 'sec-01d83hjj2yehgykzn85n2h5pa5'
YAV_VAULT_ITEM_OWNER = 'JUPYTER_CLOUD'
YAV_VAULT_ITEM_NAME = 'yav_oauth_token'


class JupyterCloudTaskMixin(object):
    class Parameters(sdk2.Task.Parameters):
        use_latest_sandbox_binary = sdk2.parameters.Bool(
            'Always use latest SANDBOX_TASKS_BINARY task resource',
            default=False,
            description='Makes sense for regular runs through Sandbox shchedulers'
        )

        with sdk2.parameters.Group('YaVault options') as ya_vault_options:
            yav_secret_id = sdk2.parameters.String(
                'YaVault secret ID',
                required=True,
                default=JUPYTERCLOUD_SECRETS_ID,
            )

            yav_oauth_token = sdk2.parameters.Vault(
                'YaVault OAuth token vault item',
                default='{}:{}'.format(YAV_VAULT_ITEM_OWNER, YAV_VAULT_ITEM_NAME),
                required=True
            )

    def _set_tasks_resource(self):
        if (
            self.Requirements.tasks_resource is not None and
            not self.Parameters.use_latest_sandbox_binary
        ):
            return

        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary \
            .find(
                state=ctr.State.READY,
                attrs=dict(
                    task_type=self.type.name,
                )
            ) \
            .first()

    def on_create(self):
        self._set_tasks_resource()

    def on_save(self):
        self._set_tasks_resource()

        if self.Requirements.tasks_resource is None:
            raise TaskFailure('Failed to find proper {} resource to run'
                              .format(sdk2.service_resources.SandboxTasksBinary.name))

    def _get_secret(self, name):
        from jupytercloud.tools.lib.environment import environment

        with environment(
            'production',
            self.Parameters.yav_oauth_token.data()
        ) as env:
            return env.get_secret(name)
