# coding: utf-8

import logging
import requests
import urllib
from datetime import date, datetime
from sandbox import sdk2

STARTREK_URL = 'https://st.yandex-team.ru'
STARTREK_API_URL = 'https://st-api.yandex-team.ru/v2'
STARTREK_API_ISSUES_URL = '/issues'
STARTREK_QUEUE = 'BIPA'
STARTREK_BOARD_ID = 14710
TELEGRAM_API_URL = 'https://api.telegram.org'
TELEGRAM_API_SEND_MESSAGE_URL = '/sendMessage'
TELEGRAM_CHAT_ID = ''  # -549454712 | -1001358205122
TELEGRAM_MESSAGE_HEAD = 'The following tickets are about to expire its SLA'


def is_sla_expiring(i):
    if 'sla' in i and i['sla'] is not None and i['sla'][0]['warnAt'] is not None:
        date_ = datetime.strptime(i['sla'][0]['warnAt'][:-5], '%Y-%m-%dT%H:%M:%S.%f')
        date_ = date(date_.year, date_.month, date_.day)

        return date_ <= date.today()

    return False


class SlaObserverTask(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):

        st_api_url = sdk2.parameters.String('Startrek API URL', default=STARTREK_API_URL, required=True)
        st_oauth_token = sdk2.parameters.YavSecret('Startrek OAuth Token', required=True)
        st_queue = sdk2.parameters.String('Startrek queue', default=STARTREK_QUEUE, required=True)
        st_board_id = sdk2.parameters.Integer('Startrek board ID', default=STARTREK_BOARD_ID, required=True)
        tg_api_url = sdk2.parameters.String('Telegram Bot API Url', default=TELEGRAM_API_URL, required=True)
        tg_bot_token = sdk2.parameters.YavSecret('Telegram Bot Token', required=True)
        tg_chat_id = sdk2.parameters.String('Telegram channel', default=TELEGRAM_CHAT_ID, required=True)

    def on_execute(self):
        query = 'Queue:{} AND "Sprint In Progress By Board":{} AND Status: !Закрыт AND Status: !Решен'.format(
            self.Parameters.st_queue,
            self.Parameters.st_board_id
        )
        url = '{}{}?query={}'.format(
            self.Parameters.st_api_url,
            STARTREK_API_ISSUES_URL,
            urllib.quote(query)
        )

        token = self.Parameters.st_oauth_token.data()['token']
        headers = {'Authorization': 'OAuth {}'.format(token),
                   'Content-Type': 'application/json',
                   'Cache-Control': 'no-cache'}
        r = requests.get(url, headers=headers, timeout=35)

        if r.status_code != requests.codes.ok or r.headers['Content-Type'] == 'application/json':
            raise Exception('Bad startrek response: {}'.format(r.content))

        issues = list(filter(is_sla_expiring, r.json()))
        items = list(map(lambda i: {
            'id': i['id'],
            'key': i['key'].replace('-', '\\-'),
            'url': '{}/{}'.format(STARTREK_URL, i['key'].replace('-', '\\-')),
            'type': i['type']['display'],
            'sla': i['sla'][0]['warnAt']
        }, issues))

        if len(items):
            text = '{}%0A{}'.format(
                TELEGRAM_MESSAGE_HEAD,
                '%0A'.join('[' + x['key'] + '](' + x['url'] + ')' for x in items)
            )
            token = self.Parameters.tg_bot_token.data()['token']
            url = '{}/bot{}{}?chat_id={}&parse_mode=MarkdownV2&text={}'.format(
                self.Parameters.tg_api_url,
                token,
                TELEGRAM_API_SEND_MESSAGE_URL,
                self.Parameters.tg_chat_id,
                text
            )

            r = requests.get(url)

            if r.status_code != requests.codes.ok:
                raise Exception('Bad telegram response: {}'.format(r.content))
        else:
            logging.info('No tickets matching the given criteria')
