import logging
import os
from json import dumps
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.misc as ctm

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.common.errors import TaskFailure


class FindSecBugsAnalyzerResult(sdk2.Resource):
    any_arch = True


class FindSecBugsAnalyzer(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [PipEnvironment('xmltodict')]
        ram = 8192
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 28800
        repo = sdk2.parameters.String('Repository name', required=True)
        branch = sdk2.parameters.String('Branch name', default_value='master')
        _container = sdk2.parameters.Container('Environment container resource', default_value=731308562, required=True)

    @property
    def env(self):
        try:
            return self._env
        except AttributeError:
            self._env = dict(l.split('=', 1) for l in
                             sp.check_output(['/bin/bash', '-c', '. /etc/profile && printenv']).splitlines())
            self._env['LANG'] = 'en_US.UTF-8'
        return self._env

    def _git_clone(self):
        folder = str(self.path('work'))
        git_cmd = 'ssh-keyscan -H github.yandex-team.ru >> ~/.ssh/known_hosts && ' \
                  'git clone --depth 1 git@github.yandex-team.ru:{repo}.git ' \
                  '-b {branch} {folder}'.format(repo=self.Parameters.repo, branch=self.Parameters.branch, folder=folder)

        try:
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('git')) as pl:
                sp.check_call(git_cmd, env=self.env, shell=True, stderr=pl.stdout, stdout=pl.stdout)
            return folder
        except sp.CalledProcessError as e:
            logging.error('Run git clone error: %s', e)
            return None

    def _run_gradlew(self, cwd):
        gradlew_cmd = 'cd {cwd} && ./gradlew assembleDebug'.format(cwd=cwd)

        for i in range(5):
            try:
                with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('gradlew attempt #{}'.format(i+1))) as pl:
                    sp.check_call(gradlew_cmd, env=self.env, shell=True, stderr=pl.stdout, stdout=pl.stdout)
                return True
            except sp.CalledProcessError as e:
                logging.error('Run gradlew script attempt %d failed: %s', i+1, e)

        return False

    @staticmethod
    def _convert_findsecbugs_result(result_xml):
        from xmltodict import parse

        result_json = parse(result_xml)
        collection = result_json.get('BugCollection', {})
        instance_list = collection.get('BugInstance', [])
        result = []

        for instance in instance_list:
            method = instance['Method']
            problem = {
                'type': instance['@type'],
                'priority': int(instance['@priority']),
                'rank': int(instance['@rank']),
                'abbrev': instance['@abbrev'],
                'classname': method.get('@classname', ''),
                'name': method.get('@name', ''),
                'path': method['SourceLine'].get('@sourcepath', ''),
                'file': method['SourceLine'].get('@sourcefile', ''),
                'start': int(method['SourceLine'].get('@start', 0)),
                'end': int(method['SourceLine'].get('@end', 0))
            }
            result.append(dict(problem=problem))

        return dict(result=result)

    def _run_findsecbugs(self, repo_path):
        cwd = '/usr/local/findsecbugs'
        result_path = str(self.path('result.xml'))
        findsecbugs_cmd = 'cd {cwd} && ' \
                          './findsecbugs.sh -high -progress ' \
                          '-xml -output {result_path} {repo_path}'.format(cwd=cwd,
                                                                          result_path=result_path, repo_path=repo_path)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('findsecbugs')) as pl:
            sp.check_call(findsecbugs_cmd, env=self.env, shell=True, stderr=pl.stdout, stdout=pl.stdout)

        with open(result_path, 'r') as f:
            findsecbugs_xml = f.read()

        findsecbugs_json = self._convert_findsecbugs_result(findsecbugs_xml)
        return findsecbugs_json

    def _save_to_res(self, result):
        results_path = str(self.path('results'))
        os.mkdir(results_path)
        results_archive_path = str(self.path('results.tar.gz'))
        result_filename = os.path.join(results_path, 'result.json')

        result = dumps(result)
        with open(result_filename, 'w') as f:
            f.write(result)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('results')) as pl:
            sp.check_call('cd {path} && tar cvzf {tgz} *'.format(path=results_path, tgz=results_archive_path),
                          shell=True, stderr=pl.stdout, stdout=pl.stdout)

        sdk2.ResourceData(
            FindSecBugsAnalyzerResult(
                self,
                'FindSecBugs results',
                results_archive_path,
            )
        )

    def on_execute(self):
        with sdk2.ssh.Key(self, self.owner, 'STOP_LEAK_SSH_KEY'):
            repo_path = self._git_clone()
        if repo_path is None:
            raise TaskFailure('Cloning git repo failed')

        gradlew_res = self._run_gradlew(repo_path)
        if not gradlew_res:
            raise TaskFailure('Running gradlew script failed')

        findsecbugs_res = self._run_findsecbugs(repo_path)
        self._save_to_res(findsecbugs_res)
