import os
import logging
import shutil
from sandbox import sdk2
from json import loads, dumps
from distutils.dir_util import copy_tree
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.market.sre.RunRenderMarketValidation import localrender


class MarketYodaxResults(sdk2.Resource):
    any_arch = True


class MarketYodax(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600
        nginx_service_conf_url = sdk2.parameters.String('Path to service part of nginx configuration', required=True)
        templates_dir = sdk2.parameters.String('Path to nginx templates directory', default='tmpl/conf/nginx',
                                               required=True)
        revision = sdk2.parameters.Integer('Revision', default=None, required=False)
        _container = sdk2.parameters.Container('Environment container resource', default_value=501902488,
                                               required=True)

    def _prepare_tree(self):
        # checkout everything from arcadia and combine valid config dir structure
        self.nginx_common_conf_url = sdk2.svn.Arcadia.ARCADIA_TRUNK_URL + '/market/sre/nanny/common/nginx/'
        self.nginx_front_common_conf_url = sdk2.svn.Arcadia.ARCADIA_TRUNK_URL + '/market/sre/nanny/front/common/nginx/'
        self.nginx_common_conf_dir = 'nginx-front-common-conf'
        self.nginx_front_common_conf_dir = 'nginx-common-conf'
        self.nginx_service_conf_dir = 'nginx-service-tmpl'
        self.templates_dir = 'templates'
        self.res_dir = 'result'
        for item in self.res_dir, self.templates_dir:
            os.mkdir(item)
        checkout_arcadia_from_url = 'arcadia:/arc/trunk/arcadia'
        if self.Parameters.revision is None:
            revision = sdk2.svn.Arcadia.parse_url(checkout_arcadia_from_url).revision
        else:
            revision = self.Parameters.revision
        sdk2.svn.Arcadia.checkout(sdk2.svn.Arcadia.ARCADIA_TRUNK_URL + '/' +
                                  self.Parameters.nginx_service_conf_url, self.nginx_service_conf_dir, revision=revision)
        sdk2.svn.Arcadia.checkout(self.nginx_common_conf_url, self.nginx_common_conf_dir, revision=revision)
        sdk2.svn.Arcadia.checkout(self.nginx_front_common_conf_url, self.nginx_front_common_conf_dir, revision=revision)
        copy_tree(os.path.join(self.nginx_common_conf_dir, 'tmpl/conf/nginx'), self.templates_dir)
        copy_tree(os.path.join(self.nginx_common_conf_dir, 'conf/nginx'), self.templates_dir)
        copy_tree(os.path.join(self.nginx_service_conf_dir, self.Parameters.templates_dir), self.templates_dir)
        copy_tree(os.path.join(self.nginx_front_common_conf_dir, 'tmpl/conf/nginx'), self.templates_dir)
        shutil.copyfile(os.path.join(self.nginx_service_conf_dir, 'tmpl/variables.tmpl'), os.path.join(self.templates_dir, 'variables.tmpl'))
        logging.info('templates dir {}'.format(os.listdir(self.templates_dir)))

    def _save_to_res(self, result):
        results_path = str(self.path('results'))
        os.mkdir(results_path)
        results_archive_path = str(self.path('results.tar.gz'))
        result_filename = os.path.join(results_path, 'result.json')

        result = dumps(result)
        with open(result_filename, 'w') as f:
            f.write(result)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('create results archive')) as pl:
            sp.check_call('cd {path} && tar cvzf {tgz} *'.format(path=results_path, tgz=results_archive_path),
                          shell=True, stderr=pl.stdout, stdout=pl.stdout)

        sdk2.ResourceData(
            MarketYodaxResults(
                self,
                'Yodax results',
                results_archive_path,
            )
        )

    def _run_yodax(self, config_path):
        yodax_cmd = 'yodax -ll -f json {}'.format(config_path)
        p = sp.Popen(yodax_cmd, shell=True, stdout=sp.PIPE, stderr=sp.PIPE)
        p.wait()
        out, err = p.communicate()

        yodax_res = loads(out)

        return yodax_res

    @staticmethod
    def _prepare_issues(unparsed_issues):
        result = []

        for issue in unparsed_issues:
            result.append(dict(problem=issue))

        return dict(result=result)

    def on_execute(self):
        self._prepare_tree()
        logging.info('Trying to render templates with mocked values')
        default_conf = os.path.join(self.templates_dir, 'sites-enabled/default.conf')
        if os.path.exists(default_conf):
            os.remove(default_conf)
        localrender.render_templates(self.templates_dir, self.res_dir)
        logging.info('Rendering succeed')

        env = 'production'

        out = os.listdir(os.path.join(self.res_dir, env, 'sites-enabled'))
        logging.info('Resulting files structure for {} is {}'.format(env, out))
        include = os.listdir(os.path.join(self.res_dir, env, 'include/hook'))
        logging.info('Resulting files structure for {} is {}'.format(env, include))

        prefix_dir = os.path.join(self.res_dir, env)
        config_path = os.path.join(prefix_dir, 'nginx.conf')

        yodax_res = self._run_yodax(config_path)
        logging.info('Yodax result for {}:\n{}\n'.format(env, yodax_res))

        result = self._prepare_issues(yodax_res)
        self._save_to_res(result)
