# coding=utf-8

from urllib import quote_plus
import os
import logging
import re
import requests

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common import file_utils as fu
from sandbox.projects.kassa import KassaAmmo
from sandbox.sdk2.vcs.git import Git
from sandbox.common import fs

kassa_back_host = 'generator.kassa.load.yandex.net:22701'
generator_url = 'http://{}/z/script'.format(kassa_back_host)


class NoUrlsFound(Exception):
    """ Exception when generator returns empty list of urls"""
    pass


class AmmoGenerator(object):
    def __init__(self, console_url, console_login, console_pwd, output='v1_ammo.txt', js_script=None):
        self.console_url = console_url
        self.console_login = console_login
        self.console_pwd = console_pwd
        self.js_script = js_script
        self.output = output

    @staticmethod
    def _read_js_script(filename):
        try:
            with open(filename, 'r') as ffile:
                data = ffile.read()
            return '='.join(['query', quote_plus(data)])
        except IOError:
            logging.exception('Can\'t find file with js script')

    @staticmethod
    def parse_response(response_data):
        """
            :param response_data: str
            :return: list of urls, can be empty
        """
        url_pattern = r'/v1\S*\n'
        pattern = re.compile(url_pattern)
        return pattern.findall(response_data)

    def get_test_data(self):

        if not self.js_script:
            dir_path = os.path.dirname(os.path.realpath(__file__))
            self.js_script = self._read_js_script(os.path.join(dir_path, 'load_script.txt'))

        request_headers = {
            'Host': kassa_back_host,
            'Content-Type': 'application/x-www-form-urlencoded',
            'Accept-Encoding': 'gzip, deflate',
            'Accept': '*/*'
        }

        try:
            logging.info('Requesting data...')
            response = requests.post(
                self.console_url,
                headers=request_headers,
                auth=(self.console_login, self.console_pwd),
                data=self.js_script
            )
            logging.debug('Response is %s', response.text)
            urls = self.parse_response(response.text)

        except requests.RequestException:
            logging.exception('Failed to get data')
            raise errors.TaskError('Request to {} failed'.format(kassa_back_host))

        if urls:
            return urls
        else:
            logging.exception('No urls found in response')


class KassaGenerateAmmo(sdk2.Task):
    """ Generates ammo for shooting on media/tickets """

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Git parameters') as git_params:
            git_repository = sdk2.parameters.Url(
                'Git repo',
                default='https://github.yandex-team.ru/media/tickets.git',
                required=True
            )
            git_branch = sdk2.parameters.String(
                label='Branch name',
                default='develop',
                required=True
            )
        with sdk2.parameters.Group('Resource parameters') as resource_params:
            use_parent_resource = sdk2.parameters.Bool(
                'Использовать родительский ресурс?',
                default=False
            )
            parent_resource_v1 = sdk2.parameters.ParentResource(
                'Ammo for v1 urls',
                resource_type=KassaAmmo,
                required=False
            )
            parent_resource_v2 = sdk2.parameters.ParentResource(
                'Ammo for v2 urls',
                resource_type=KassaAmmo,
                required=False
            )

        with sdk2.parameters.Output:
            result_ready_v1 = sdk2.parameters.Bool(
                'Ammo for v1 is ready', default=False, required=True
            )
            result_ready_v2 = sdk2.parameters.Bool(
                'Ammo for v2 is ready', default=False, required=True
            )

    def clone_repository(self):
        directory = self.path('repository').as_posix()
        Git(self.Parameters.git_repository).clone(directory, self.Parameters.git_branch)
        return directory

    def on_execute(self):
        repository_directory = self.clone_repository()
        logging.info('Repository cloned')

        v2_ammo_urls = fu.read_lines(os.path.join(repository_directory, 'load', 'orders_ammo.txt'))
        if self.Parameters.use_parent_resource:
            resource_v2_data = sdk2.ResourceData(self.Parameters.parent_resource_v2)
            logging.debug('Use parent resource for v2')
        else:
            resource_v2 = KassaAmmo(self, 'Kassa ammo v2 urls', 'v2_ammo.txt')
            resource_v2.urls_type = 'v2'
            resource_v2_data = sdk2.ResourceData(resource_v2)
            logging.debug('Using independent resource for v2')
        fu.write_lines(resource_v2_data.path.as_posix(), v2_ammo_urls)
        # noinspection PyCompatibility
        fs.allocate_file(
            unicode(self.Parameters.parent_resource_v2.path),
            os.path.getsize(str(self.Parameters.parent_resource_v2.path))
        )
        self.Parameters.result_ready_v2 = True

        load_script = fu.read_file(os.path.join(repository_directory, 'load', 'load_script.txt'))
        encoded_script = 'query=' + quote_plus(load_script)
        logging.debug('Script is %s...', encoded_script)

        generator = AmmoGenerator(
            console_url=generator_url,
            console_login='test',
            console_pwd='test',
            js_script=encoded_script,
            output=''
        )
        ammo_urls = generator.get_test_data()
        logging.debug('Urls are %s', ammo_urls)
        if ammo_urls:
            if self.Parameters.use_parent_resource:
                resource_v1_data = sdk2.ResourceData(self.Parameters.parent_resource_v1)
                logging.debug('Use parent resource for v1')
            else:
                resource_v1 = KassaAmmo(self, 'Kassa ammo v1 urls', 'v1_ammo.txt')
                resource_v1.urls_type = 'v1'
                resource_v1_data = sdk2.ResourceData(resource_v1)
                logging.debug('Using independent resource for v1')
            fu.write_lines(resource_v1_data.path.as_posix(), ammo_urls)
            # noinspection PyCompatibility
            fs.allocate_file(
                unicode(self.Parameters.parent_resource_v2.path),
                os.path.getsize(str(self.Parameters.parent_resource_v2.path))
            )
            self.Parameters.result_ready_v1 = True
        else:
            logging.info('No urls found in response')
            self.Parameters.result_ready_v1 = False
