# -*- coding: utf-8 -*-

import logging
import logging.handlers
import time
import re
import os

from sandbox import sdk2
from sandbox.projects.kassa import KassaAmmo
from sandbox.projects.kassa.generate_ammo import KassaGenerateAmmo
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.projects.tank.LoadTestResults import LoadTestResults
from sandbox import common
from sandbox.common import errors
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt

TANKS_SAS = [
    'stuart.tanks.yandex.net:8083',
    'chaffee.tanks.yandex.net:8083',
    'hellcat.tanks.yandex.net:8083',
    'vickers.tanks.yandex.net:8083',
    'lee.tanks.yandex.net:8083',
    'wolverine.tanks.yandex.net:8083',
    'scorpion.tanks.yandex.net:8083'
]


def logger():
    loggerr = logging.getLogger('%s_%s' % (__name__, time.time()))
    loggerr.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s %(levelname)s [%(processName)s: %(threadName)s] %(message)s')
    file_handler = logging.handlers.RotatingFileHandler(
        'kassa_metashooting.log',
        maxBytes=1024 * 1024,
        backupCount=5
    )

    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(formatter)
    loggerr.addHandler(file_handler)
    return loggerr


def read_file(key):
    if common.system.inside_the_binary():
        from library.python import resource
        key = os.path.join("sandbox/projects/kassa", key)
        resource_data = resource.find(key)
        if resource_data is not None:
            logging.debug("Arcadia resource %r loaded", key)
            return resource_data

    kassa_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
    config_path = os.path.join(kassa_dir, key)
    with open(config_path, 'r') as fileobj:
        return fileobj.read()


v1_config = read_file('config/v1.tank.ini')
v2_config = read_file('config/v2.tank.ini')


class KassaMetaShooting(sdk2.Task):
    """ Task for shooting on Kassa (media/tickets), cf. st/KASSA-886 """

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Task parameters'):
            version = sdk2.parameters.String(
                'Version of object under test',
            )
            ticket = sdk2.parameters.String(
                'ST ticket for shooting',
                description='The shoot will be performed in this ticket. The report will be sent as a comment',
                default='KASSA-886',
                required=True
            )
            token = sdk2.parameters.String(
                'ST token for comment',
                description='Task owner must have permissions to access this token in SB Vault.',
                default='lunapark-startrek-token'
            )

        with sdk2.parameters.Group('Ammo parameters'):
            generate_fresh_ammo = sdk2.parameters.Bool(
                'Generate fresh ammo?',
                description='Whether fresh ammo is generated or the latest ammo is taken',
                default=True
            )
            with generate_fresh_ammo.value[True]:
                with sdk2.parameters.Group('Ammo generator settings'):
                    git_repo = sdk2.parameters.String(
                        label=KassaGenerateAmmo.Parameters.git_repository.name,
                        default_value=KassaGenerateAmmo.Parameters.git_repository.default_value
                    )
                    git_branch = sdk2.parameters.String(
                        label=KassaGenerateAmmo.Parameters.git_branch.name,
                        default_value=KassaGenerateAmmo.Parameters.git_branch.default_value
                    )

    class Context(sdk2.Task.Context):
        shoot_data = {
            'v1': {
                'lunapark_link': '',
                'desc': 'Shoot on KASSA v1 ammo',
                'config_content': v1_config,
                'ammo': None,
                'subtask': None
            },
            'v2': {
                'lunapark_link': '',
                'desc': 'Shoot on KASSA v2 ammo',
                'config_content': v2_config,
                'ammo': None,
                'subtask': None
            }
        }
        report_tasks = []

    def start_shooting(self, ammo_type):
        shoot = self.Context.shoot_data[ammo_type]
        subtask_shoot = ShootViaTankapi(
            self,
            description=shoot['desc'],
            ammo_source='resource',
            ammo_resource=shoot['ammo'],
            use_monitoring=True,
            monitoring_source='resource',
            monitoring_resource=412573927,
            config_source='file',
            config_content=shoot['config_content'],
            tanks=TANKS_SAS
        ).enqueue()
        logger().info('Subtask with shooting %s is started', ammo_type)
        shoot['subtask'] = subtask_shoot.id
        raise sdk2.WaitTask([subtask_shoot.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True,
                            timeout=14400)

    def start_reporting(self, ammo_type):
        shoot = self.Context.shoot_data[ammo_type]
        logger().info('Start reporting for %s, shoot %s (task %s)',
                      ammo_type, shoot['lunapark_link'], shoot['subtask'])
        messages = {
            'v1': 'Патроны /v1/cities и /v1/events',
            'v2': 'Патроны /v2/orders и /v2/user/profile',
        }

        if shoot['lunapark_link']:
            logger().info('Lunapark link for %s shoot is %s', ammo_type, shoot['lunapark_link'])
            shoot_id = re.findall(r'\d+', shoot['lunapark_link'])[0]
            subtask_report = LoadTestResults(
                self,
                shoot_id=shoot_id,
                report_type='const',
                additional_message=messages[ammo_type],
                send_comment=True,
                ticket_id=self.Parameters.ticket,
                st_token_name=self.Parameters.token,
            ).enqueue()
            self.Context.report_tasks.append(subtask_report)

    def on_execute(self):

        def take_freshest_ammo(urls_type):
            resource = KassaAmmo.find(
                state=ctr.State.READY,
                attrs=dict(urls_type=urls_type)
            ).order(-sdk2.Resource.id).first()
            if not resource:
                raise errors.TaskError('No ammo found for %s urls_type', urls_type)
            else:
                sdk2.ResourceData(resource)
                return resource

        # Step 1: Generate ammo
        with self.memoize_stage.generate_ammo:
            if self.Parameters.generate_fresh_ammo:
                ammo_generation_subtask = KassaGenerateAmmo(
                    self,
                    description='Generate ammo for task {}'.format(self.id),
                    owner=self.owner,
                    git_repository=self.Parameters.git_repo,
                    git_branch=self.Parameters.git_branch,
                ).enqueue()
                ammo_ready = {ammo_generation_subtask.id: ('result_ready_v1', 'result_ready_v2')}
                raise sdk2.WaitOutput(ammo_ready, wait_all=True)

        for shoot_type, shoot_data in self.Context.shoot_data.items():
            shoot_data['ammo'] = take_freshest_ammo(shoot_type).id
            logger().info('Ammo with %s urls is %s', shoot_type, shoot_data['ammo'])

        # Step 2: Run shooting
        for ammo_type in self.Context.shoot_data.keys():
            shoot = self.Context.shoot_data[ammo_type]
            with self.memoize_stage['shoot_{}'.format(ammo_type)]:
                self.start_shooting(ammo_type)
            if shoot['subtask']:
                shoot['lunapark_link'] = sdk2.Task[shoot['subtask']].Parameters.lunapark_link

        # Step 3: Make a report
        with self.memoize_stage.reporting:
            self.start_reporting('v1')
            self.start_reporting('v2')
            if self.Context.report_tasks:
                raise sdk2.WaitTask(
                    self.Context.report_tasks,
                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                    wait_all=True
                )
