import logging
import re
import requests

import sandbox.sdk2 as sdk2


TODAY = 'today'


class EmptyClidException(Exception):
    pass


class KeyboardClidSearch(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):


        with sdk2.parameters.Output():
            clids_output = sdk2.parameters.String('results', multiline=True)

        with sdk2.parameters.Group('Auth params', collapse=True) as auth_params:
            ch_user = sdk2.parameters.String(
                'clickhouse_user',
                default_value='robot-keyboard',
                required=True
            )
            ch_secret = sdk2.parameters.YavSecret(
                'ch_secret',
                default='sec-01eskjgxb9jx2s8y88086z7pfy'
            )

        with sdk2.parameters.Group('Distribution params', collapse=True) as distribution_group:
            api_key = sdk2.parameters.Integer(
                'api_key',
                default_value=112684,
                required=True,
            )
            device_id = sdk2.parameters.String(
                'device_id',
                required=True
            )
            date = sdk2.parameters.String(
                'date',
                required=True,
                default_value=TODAY,
                description='date format: YYYY-MM-DD or today'
            )
            clids = sdk2.parameters.Dict(
                'clids',
                required=True
            )

    def on_execute(self):
        super(KeyboardClidSearch, self).on_execute()
        data = self.get_metrika_results(self.get_query()).strip()
        if len(data.strip()) == 0:
            raise EmptyClidException('There are no clids!')
        else:
            logging.info('Clids found')
            logging.info(data)
            self.Parameters.clids_output = data

    def get_metrika_results(self, query):
        return requests.post(
            'http://clickhouse.metrika.yandex.net:8123/',
            auth = (
                self.Parameters.ch_user,
                self.Parameters.ch_secret.data()['metrika-robot-password']
            ),
            params = {'query': query},
            timeout = 2000
        ).text

    def get_query(self, **kwargs):
        template = 'select {} from mobgiga.events_all where {} order by EventTimestamp limit 100'
        selection = self.get_selection()
        where = self.get_condition()
        query = template.format(selection, where).strip().replace('\n', ' ')
        query = re.sub(r'\s+', ' ', query)
        logging.info('query: {}'.format(query))
        return query

    def get_selection(self):
        return '''
            Clids.Names,
            Clids.Values,
            EventDate,
            EventTimestamp
        '''

    def get_condition(self):
        where = '''
            APIKey = {}
            AND DeviceID = '{}'
            AND EventDate = {}

        '''
        clids = self.Parameters.clids.items()
        for clid_name, clid_value in clids:
            where += '''
            AND has(Clids.Names, '{}')
            AND has(Clids.Values, {})
            '''.format(clid_name, clid_value)

        return where.format(self.Parameters.api_key, self.Parameters.device_id, self._get_date())

    def _get_date(self):
        date = self.Parameters.date
        if date == TODAY:
            return 'today()'
        else:
            return "toDate('{}')".format(date.strip())
