import json
import requests
import shutil
import tempfile
import zipfile

from sandbox.common import errors

import sandbox.sdk2 as sdk2
import sandbox.projects.resource_types as resource_types


class KeyboardPackage(sdk2.Resource):
    ttl = 180
    auto_backup = True

    platform = sdk2.parameters.String('Platform these binaries were built for: Android/iOS')
    commit_time = sdk2.parameters.String('Time of commit to arcadia this binaries were built from.')
    svn_revision = sdk2.parameters.String('Svn revision this binaries were built from.')


class KeyboardPackageAnalysisReport(sdk2.Resource):
    """JSON with keyboard package size analysis report"""


class KeyboardPackageSizeAnalyze(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        keyboard_package = sdk2.parameters.Resource(
            "Keyboard Package with native libs and resources",
            resource_type=KeyboardPackage,
            required=True
        )

        with sdk2.parameters.Output:
            report = sdk2.parameters.Resource(
                "Package size analysis report",
                resource_type=KeyboardPackageAnalysisReport,
            )

    def on_execute(self):
        super(KeyboardPackageSizeAnalyze, self).on_execute()

        keyboard_package_resource = self.Parameters.keyboard_package
        keyboard_package = sdk2.ResourceData(keyboard_package_resource)

        report_resource = KeyboardPackageAnalysisReport(
            self,
            "Package size report for sbr:{}".format(keyboard_package_resource.id),
            'report.json',
        )
        report = sdk2.ResourceData(report_resource)

        self.Parameters.report = report_resource.id

        root_path = tempfile.mkdtemp()
        try:
            with zipfile.ZipFile(str(keyboard_package.path), "r") as zip_ref:
                zip_ref.extractall(root_path)

            self._build_report(root_path, str(report.path))
            report.ready()
        except:
            raise
        finally:
            shutil.rmtree(root_path)

    def _build_report(self, root_path, report_path):
        import smart_devices.ci.pylibs.package_size.analyze as A
        import smart_devices.ci.pylibs.package_size.tree as T

        report_data = A.build_report(
            root_path=root_path,
            maps_root_path=root_path,  # dummy path
        )

        stop_at = {'file'}
        T.prune_file_tree(report_data['tree'], stop_at)

        meta_path = root_path + '/res/raw/meta.json'
        with open(meta_path) as f:
            report_data['meta'] = json.load(f)

        self.Context.metrics = report_data['metrics']
        self.Context.save()

        with open(report_path, 'w') as f:
            f.write(json.dumps(
                report_data,
                indent=2,
                sort_keys=True,
            ))
            f.write('\n')


class KeyboardPackageSizeDiff(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        report_a = sdk2.parameters.Resource(
            "First (older) package size analysis report",
            resource_type=KeyboardPackageAnalysisReport,
            required=True,
        )
        report_b = sdk2.parameters.Resource(
            "Second (newer) package size analysis report",
            resource_type=KeyboardPackageAnalysisReport,
            required=True,
        )

        diff_threshold = sdk2.parameters.Integer("Max allowable diff size", default=200 * 1024)

        pull_request_id = sdk2.parameters.Integer("Pull request id")

        fail_on_diff_threshold = sdk2.parameters.Bool(
            "Fail on too large diff",
            default=True
        )

        report_on_diff_threshold = sdk2.parameters.Bool(
            "Report to PR detailed information about diff",
            default=False
        )

        secret = sdk2.parameters.YavSecret("YAV secret token")

        with sdk2.parameters.Output:
            diff_report = sdk2.parameters.Resource(
                "Diff report",
                resource_type=resource_types.PLAIN_TEXT,
            )

    def on_execute(self):
        import smart_devices.ci.pylibs.package_size.diff as D
        import smart_devices.ci.pylibs.package_size.tree as T

        super(KeyboardPackageSizeDiff, self).on_execute()

        report_a = sdk2.ResourceData(self.Parameters.report_a)
        with open(str(report_a.path), 'r') as f:
            report_a_data = json.loads(f.read())
            dump_a = D.PackageDump(report_a_data)

        report_b = sdk2.ResourceData(self.Parameters.report_b)
        with open(str(report_b.path), 'r') as f:
            report_b_data = json.loads(f.read())
            dump_b = D.PackageDump(report_b_data)

        self._report_lines = []

        def add_message(line):
            self.set_info(line)
            self._report_lines.append(line)

        diff = D.PackageSizeDiff(report_a_data, report_b_data)
        has_diff, diff_msg = diff.check_diff(self.Parameters.diff_threshold)

        add_message(diff_msg)

        self.Context.has_diff = has_diff
        self.Context.save()

        diff_tree = diff.diff_tree()
        if diff_tree['has_diff']:
            msg = '\n'.join(
                ['Detailed diff:'] + list(T.iter_text_diff_tree(diff_tree))
            )
            add_message(msg)

        diff_report_resource = resource_types.PLAIN_TEXT(self, "Diff report", "diff_report.txt")
        diff_report = sdk2.ResourceData(diff_report_resource)
        with open(str(diff_report.path), 'w') as f:
            f.write('\n'.join(self._report_lines))
        diff_report.ready()

        self.Parameters.diff_report = diff_report_resource.id

        if has_diff:
            if self.Parameters.report_on_diff_threshold and self.Parameters.pull_request_id > 0:
                self._post_comment()

            if self.Parameters.fail_on_diff_threshold and dump_a.total_size < dump_b.total_size:
                raise errors.TaskFailure("Failed: too large diff size.")

    def _post_comment(self):
        arcanum_token = self.Parameters.secret.data()['arcanum-token']
        comment = '**Too large diff:**\n'
        quote = '```'
        comment += '\n'.join([quote] + self._report_lines + [quote])

        res = requests.post(
            "https://a.yandex-team.ru/api/v1/review-requests/%d/comments" % self.Parameters.pull_request_id,
            json={'content': comment},
            headers={'Authorization': 'OAuth %s' % arcanum_token}
        )
        res.raise_for_status()
