# -*- coding: UTF-8 -*-

import json
import os.path
import shutil

import sandbox.common as common
from sandbox import sdk2
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.projects.common.build import YaPackage
from sandbox.projects.common.build import parameters as pcb_parameters
from sandbox.projects.common.constants import CHECKOUT_MODE_AUTO
from sandbox.projects.common.nanny import nanny
from sandbox.projects.kikimr.resources import KikimrPackage


class KikimrBuildYaPackage(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Parameters(sdk2.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl("Arcadia url", required=True)
        do_not_remove = sdk2.parameters.Bool("Set 'do not remove' for resources", default_value=False)
        startrek_tickets = sdk2.parameters.String("Startrek ticket ids", default_value="", required=False)
        auto_release = sdk2.parameters.Bool("Nanny auto release", default_value=False)
        with auto_release.value[True]:
            with sdk2.parameters.String("Release Status", multiline=True, required=True) as nanny_status:
                nanny_status.values[ctt.ReleaseStatus.STABLE] = None
                nanny_status.values[ctt.ReleaseStatus.PRESTABLE] = None
                nanny_status.values[ctt.ReleaseStatus.TESTING] = None
                nanny_status.values[ctt.ReleaseStatus.UNSTABLE] = nanny_status.Value(default=True)
            nanny_subject = sdk2.parameters.String("Release Subject", default_value="")
            nanny_comments = sdk2.parameters.String("Release Comments", default_value="")
            nanny_to = sdk2.parameters.List("To", sdk2.parameters.String)
            nanny_cc = sdk2.parameters.List("Cc", sdk2.parameters.String)

    class Requirements(sdk2.Requirements):
        disk_space = 25 * 1024  # 25Gb

    class CustomParameters(object):
        package_type = YaPackage.TARBALL
        ttl = 30
        package_path = ""
        host_platform = None
        target_platform = None
        output_resource = KikimrPackage.name
        publish_package = False
        publish_to = ""
        key_user = None
        package_name = None
        package_description = None
        comment = None
        resource_attributes = {
            "branch",
            "build_type",
            "platform",
            "resource_name",
            "resource_version",
            "svn_path",
            "svn_revision",
        }

    def start_build(self):
        input_parameters = {
            "description": self.Context.description,
            pcb_parameters.ArcadiaUrl.name: self.Parameters.arcadia_url,
            YaPackage.UseNewFormatParameter.name: True,
            YaPackage.PackagesParameter.name: self.CustomParameters.package_path,
            YaPackage.PackageTypeParameter.name: self.CustomParameters.package_type,
            YaPackage.PublishPackageParameter.name: self.CustomParameters.publish_package,
            YaPackage.PublishToParameter.name: self.CustomParameters.publish_to,
            pcb_parameters.CheckoutModeParameter.name: CHECKOUT_MODE_AUTO,
        }
        if self.CustomParameters.key_user is not None:
            input_parameters[YaPackage.KeyUserParameter.name] = self.CustomParameters.key_user
        if self.CustomParameters.output_resource is not None:
            input_parameters[YaPackage.ResourceTypeParameter.name] = self.CustomParameters.output_resource
        if self.CustomParameters.host_platform is not None:
            input_parameters[YaPackage.HostPlatformParameter.name] = self.CustomParameters.host_platform
        if self.CustomParameters.target_platform is not None:
            input_parameters[YaPackage.TargetPlatformParameter.name] = self.CustomParameters.target_platform

        build_task = YaPackage(self, **input_parameters)
        build_task.Requirements.disk_space = self.Requirements.disk_space
        build_task.save().enqueue()
        self.Context.build_task_id = build_task.id
        raise sdk2.WaitTask(
            build_task,
            list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
            wait_all=True
        )

    def verify_subtask(self):
        task = self.find(None, id=self.Context.build_task_id).first()
        if task.status in ctt.Status.Group.BREAK + ctt.Status.Group.SCHEDULER_FAILURE:
            raise common.errors.TaskError("Subtask unsuccessfully finished with status " + str(task.status))
        return task

    def get_resource_attributes(self, resource):
        attributes = dict()
        for name in self.CustomParameters.resource_attributes:
            if hasattr(resource, name):
                attributes[name] = getattr(resource, name)
        self.Context.resource_revision = attributes.get("svn_revision")
        return attributes

    def copy_resource(self, task):
        resource = sdk2.Resource[self.CustomParameters.output_resource].find(task=task).first()
        if resource is None:
            raise common.errors.TaskError("Subtask resource not found")
        src_data = sdk2.ResourceData(resource)
        src_path = str(src_data.path.absolute())
        resource_name = os.path.basename(src_path)

        dst_resource = sdk2.Resource[self.CustomParameters.output_resource](
            self,
            self.Context.description,
            resource_name,
            ttl="inf" if self.Parameters.do_not_remove else self.CustomParameters.ttl,
            **self.get_resource_attributes(resource)
        )
        dst_data = sdk2.ResourceData(dst_resource)
        dst_path = str(dst_data.path.absolute())
        if os.path.isfile(src_path):
            shutil.copy(src_path, dst_path)
        else:
            shutil.copytree(src_path, dst_path)
        dst_data.ready()

    def get_package_info(self):
        url_info = sdk2.svn.Arcadia.parse_url(self.Parameters.arcadia_url)
        package_url = sdk2.svn.Arcadia.replace(
            self.Parameters.arcadia_url,
            path=os.path.join(url_info.path, os.path.dirname(self.CustomParameters.package_path))
        )
        package_filename = os.path.basename(self.CustomParameters.package_path)
        package_svndir = sdk2.svn.Arcadia.get_arcadia_src_dir(package_url)
        package_path = os.path.join(str(package_svndir), package_filename)
        try:
            with open(package_path, "r") as fd:
                package = json.load(fd)
        except IOError:
            self.set_info("Warning: Can't read package file")
            return None, None
        except ValueError:
            self.set_info("Warning: Invalid package file")
            return None, None
        return package.get("meta", {}).get("name"), package.get("meta", {}).get("description")

    def update_description(self):
        self.Context.startrek_ticket_ids = nanny.StartrekTicketIdsParameter.cast(self.Parameters.startrek_tickets)
        package_name = self.CustomParameters.package_name
        package_description = self.CustomParameters.package_description
        if package_name is None or package_description is None:
            pkg_name, pkg_descr = self.get_package_info()
            if package_name is None:
                package_name = pkg_name
            if package_description is None:
                package_description = pkg_descr

        description = list()
        if package_description is not None:
            description.append(package_description)
        if package_name is not None:
            description.append("({})".format(package_name))
        if self.CustomParameters.comment is not None:
            description.append(self.CustomParameters.comment)
        self.Context.description = " ".join(description)

    def release_subject(self):
        subject = self.Parameters.nanny_subject
        if subject is not None and subject != "":
            return subject
        subject = "Release {res} ({task})".format(res=self.CustomParameters.output_resource, task=self.type)
        if self.Context.resource_revision != ctm.NotExists and \
                self.Context.resource_revision is not None:
            subject += " r{}".format(self.Context.resource_revision)
        return subject

    def on_execute(self):
        with self.memoize_stage.update_description:
            self.update_description()

        with self.memoize_stage.start_build:
            self.start_build()

        task = self.verify_subtask()
        self.copy_resource(task)

    def on_success(self, prev_status):
        super(KikimrBuildYaPackage, self).on_success(prev_status)
        if not self.Parameters.auto_release:
            return
        nanny.ReleaseToNannyTask2.on_release(self, dict(
            releaser=self.author,
            release_status=self.Parameters.nanny_status,
            release_subject=self.release_subject(),
            email_notifications=dict(to=self.Parameters.nanny_to, cc=self.Parameters.nanny_cc),
            release_comments=self.Parameters.nanny_comments,
        ))
