# -*- coding: UTF-8 -*-

import json
import logging
import os.path
import re

import requests

import sandbox.common.types.task as ctt
from sandbox import common
from sandbox import sdk2
from sandbox.projects.cloud.platform.CloudBuildImageWithPacker import CloudBuildImageWithPacker
from sandbox.projects.kikimr.common.retry import retry
from sandbox.projects.kikimr.resources import KikimrPackerStaffInfo


class KikimrBuildCloudImage(sdk2.Task):
    """
    Build Kikimr Cloud Image
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 2 * 1024  # 2Gb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Caches(sdk2.Requirements.Caches):
        pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        description = "Build Kikimr Cloud Image"
        kill_timeout = 2 * 60 * 60  # 2 hours

        packages = sdk2.parameters.Dict("Packages")

        use_z2 = sdk2.parameters.Bool("Get package versions from Z2", default_value=False)
        with use_z2.value[True]:
            with sdk2.parameters.Group("Z2 Settings") as z2_block:
                z2_config = sdk2.parameters.String("Z2 Config Name", required=True)
                z2_api_key_vault = sdk2.parameters.String("Vault name with Z2 API Key", required=True)

        with sdk2.parameters.Group("Users syncing settings") as usersync_block:
            staff_oauth_vault = sdk2.parameters.String("Vault name with Staff OAuth token", required=True)
            staff_groups = sdk2.parameters.List("Staff groups name to sync keys", required=True)

        with sdk2.parameters.Group("Secrets settings") as secrets_block:
            secrets = sdk2.parameters.Dict("Secrets (file_path: vault_owner:vault_key)")

        with sdk2.parameters.Group("Packer Settings") as packer_block:
            template_url = sdk2.parameters.SvnUrl("Svn url of Packer template file", required=True)
            yc_folder_id = sdk2.parameters.String("YC FolderID", required=True)
            yc_subnet_id = sdk2.parameters.String("YC SubnetID", required=True)
            yc_endpoint = sdk2.parameters.String(
                "YC API Endpoint",
                required=True,
                default_value="api.cloud-preprod.yandex.net:443"
            )
            yc_token = sdk2.parameters.String("YC Token (vault_owner:vault_key)", required=False)
            yc_zone = sdk2.parameters.String("YC Zone", required=True, default_value="ru-central1-c")
            source_image_id = sdk2.parameters.String("Source ImageID", required=True)
            image_disk_size = sdk2.parameters.Integer("Image disk size (in Gb)", required=True, default=20)
            image_family = sdk2.parameters.String("Image Family", required=True, default_value="ydb-test-base")
            service_account_key = sdk2.parameters.String(
                "Service account private key (json file with RSA keypair generated by IAM) in form vault_owner:vault_key",
                required=False
            )

    class Context(sdk2.Task.Context):
        package_versions = dict()
        build_task = None
        staff_info_resource_id = None

    def z2_items(self):
        request = dict(
            apiKey=sdk2.Vault.data(self.Parameters.z2_api_key_vault),
            configId=self.Parameters.z2_config
        )
        response = self.z2_request("items", request)
        if not response.get("success", False):
            raise common.errors.TaskError("Z2 api error response " + str(response.get("errorMsg")))
        try:
            return dict((item["name"], item["version"]) for item in response["response"]["items"])
        except KeyError:
            raise common.errors.TaskError("Z2 api invalid response")

    @retry(Exception)
    def z2_request(self, action, request):
        url = "https://z2.yandex-team.ru/api/v1/" + action
        logging.info("Z2 API action %s url %s request %s", action, url, self.z2_dump_request(request))
        try:
            response = requests.get(url, params=request)
        except Exception as e:
            logging.error("Error http request %s %r", url, e)
            raise common.errors.TaskError("Z2 api http request error")
        if response.status_code != 200:
            logging.error("Z2 api response code %r %r", response.status_code, response.text)
            raise common.errors.TaskError("Z2 api http response code " + str(response.status_code))
        try:
            data = response.json()
        except Exception as e:
            logging.error("Error in Z2 API response %r %r", e, response)
            raise common.errors.TaskError("Error in Z2 API response")
        logging.info("Z2 response %r", data)
        return data

    @staticmethod
    def z2_dump_request(data):
        data_copy = data
        if isinstance(data, dict):
            data_copy = data.copy()
            if "apiKey" in data_copy:
                data_copy["apiKey"] = "HIDDEN"
        return json.dumps(data_copy, sort_keys=True, indent=4, separators=(',',  ': '))

    def get_versions_from_z2(self):
        versions = self.z2_items()
        logging.info("Z2 packages %r", versions)
        for name, ver in self.Context.package_versions.iteritems():
            if not ver and name in versions:
                self.Context.package_versions[name] = versions[name]
        self.Context.save()

    def get_versions(self):
        self.Context.package_versions = self.Parameters.packages.copy()
        self.Context.save()

    def get_package_list(self):
        package_list = list()
        for name, ver in self.Context.package_versions.iteritems():
            if ver:
                package_list.append(name + "=" + ver)
            else:
                package_list.append(name)
        return package_list

    def show_versions(self):
        msg = ["Package versions:"] + self.get_package_list()
        self.set_info("\n".join(msg))

    def build_image(self):
        svn_dir_url, template_file = os.path.split(self.Parameters.template_url)
        parameters = dict(
            kill_timeout=self.Parameters.kill_timeout - 5 * 60,
            svn_dir_url=svn_dir_url,
            template_file=template_file,
            env_vars=dict(
                YC_FOLDER_ID=self.Parameters.yc_folder_id,
                YC_SUBNET_ID=self.Parameters.yc_subnet_id,
                YC_ENDPOINT=self.Parameters.yc_endpoint,
                YC_ZONE=self.Parameters.yc_zone,
                YC_SOURCE_IMAGE_ID=self.Parameters.source_image_id,
                YC_IMAGE_DISK_SIZE=str(self.Parameters.image_disk_size),
                YC_IMAGE_FAMILY=self.Parameters.image_family,
                PACKAGES_LIST=" ".join(self.get_package_list())
            ),
            log_packer_to_file=True,
            debug_mode=False,
            on_error_mode="cleanup",
            extra_resources={
                "stuff/staff-info.json": self.Context.staff_info_resource_id
            },
            extra_secrets=self.Parameters.secrets,
        )
        if self.Parameters.yc_token:
            parameters["vault_env"] = "YC_TOKEN={}".format(self.Parameters.yc_token)
        if self.Parameters.service_account_key:
            parameters["service_account_key"] = "service_account_key.json={}".format(self.Parameters.service_account_key)
            parameters["env_vars"]["YC_SA_KEY_FILE"] = "service_account_key.json"
        build_task = CloudBuildImageWithPacker(
            self,
            owner=self.owner,
            priority=self.Parameters.priority,
            description="Build YDB image",
            **parameters
        )
        build_task.save().enqueue()
        self.Context.build_task = build_task.id
        self.Context.save()

    def wait_task(self):
        if self.Context.build_task is None:
            return
        task = sdk2.Task[self.Context.build_task].reload()
        logging.info("Task %r status is %r", task.id, task.status)
        if task.status in ctt.Status.Group.BREAK + ctt.Status.Group.SCHEDULER_FAILURE:
            raise common.errors.TaskError("Subtask id {} failed".format(task.id))
        if task.status not in ctt.Status.Group.SUCCEED:
            raise sdk2.WaitTask([task], list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK), wait_all=True)

    @retry(Exception)
    def get_staff_info(self, staff_group):
        if not re.match(r'^[a-zA-Z0-9_-]+$', staff_group):
            raise common.errors.TaskError("Incorrect staff group name")

        params = {
            "_query": "department_group.ancestors.url==\"{group}\" or department_group.url==\"{group}\"".format(
                group=staff_group
            ),
            "official.is_dismissed": "false",
            "official.is_robot": "false",
            "_limit": "500",
            "_fields": "login,keys,id,environment.shell,name.first.en,name.last.en",
        }
        url = "https://staff-api.yandex-team.ru/v3/persons"

        auth = "OAuth " + sdk2.Vault.data(self.Parameters.staff_oauth_vault)

        logging.info("Staff request params %r", params)
        try:
            response = requests.get(
                "https://staff-api.yandex-team.ru/v3/persons",
                params=params,
                headers=dict(Authorization=auth)
            )
        except Exception as e:
            logging.error("Error http request to staff api %s %r", url, e)
            raise common.errors.TaskError("Error http request to staff api")
        if response.status_code != 200:
            logging.error("Staff api response code is %r %r", response.status_code, response.text)
            raise common.errors.TaskError("Staff api response code is " + str(response.status_code))
        try:
            data = response.json()
        except Exception as e:
            logging.error("Error decode staff api response %r %r", e, response)
            raise common.errors.TaskError("Error decode staff api response")
        logging.info("Staff api response is %r", data)
        return data

    def save_staff_info(self, staff_info):
        resource = KikimrPackerStaffInfo(self, "YDB staff info", "staff-info.json")
        resource_data = sdk2.ResourceData(resource)
        dst = str(resource_data.path.absolute())
        with open(dst, "w+") as fd:
            json.dump(staff_info, fd, sort_keys=True, indent=4, separators=(',', ': '))
        resource_data.ready()
        self.Context.staff_info_resource_id = resource.id

    def make_staff_info(self, data):
        logging.info("Staff data %r", data)
        staff_info = dict()
        for item in data["result"]:
            login = item["login"]
            shell = item["environment"]["shell"]
            uid = int(item["id"]) + 20000
            name = "{first} {last}".format(
                first=item["name"]["first"]["en"],
                last=item["name"]["last"]["en"],
            )
            keys = [key_item["key"] for key_item in item["keys"]]
            if len(keys) == 0:
                logging.warning("Login %s has no keys - skip", login)
                continue
            staff_info[login] = dict(
                shell=shell,
                uid=uid,
                name=name,
                keys=keys,
            )
        logging.info("Staff info %r", staff_info)
        return staff_info

    def show_packer_info(self):
        query = sdk2.Resource.find(task=sdk2.Task[self.Context.build_task], type="TASK_LOGS").limit(1)
        task_logs_resources = list(query)
        if not task_logs_resources:
            logging.error("Can't find build task logs")
            return
        resource = task_logs_resources[0]
        logging.info("Task logs resource id %r", resource.id)
        logs_path = str(sdk2.ResourceData(resource).path)
        log_path = os.path.join(logs_path, "packer_build.out.txt")
        if not os.path.isfile(log_path):
            logging.error("Can't find packer log file %r", log_path)
            return
        try:
            with open(log_path, "r") as fd:
                fd.seek(0, 2)  # Seek to end
                file_size = fd.tell()
                if file_size == 0:
                    logging.error("Packer build log is empty")
                    return
                if file_size < 4096:
                    fd.seek(0, 0)
                else:
                    fd.seek(-4096, 2)
                lines = fd.readlines()
                if not lines:
                    logging.info("Packer build log has no lines")
                    return
                last_line = lines[-1].strip()
                if not last_line.startswith("--> yandex: A disk image was created"):
                    logging.error("Unknown build log line %r", last_line)
                    return
                self.set_info(last_line)
        except IOError as e:
            logging.error("Can't read packer build log %r", e)
            return

    def on_execute(self):
        if (self.Parameters.service_account_key and self.Parameters.yc_token) or \
           not (self.Parameters.service_account_key or self.Parameters.yc_token):
            raise common.errors.TaskError("Only one auth method must be specified (service_account_key or yc_token)")
        with self.memoize_stage.update_ssh_keys(commit_on_entrance=False):
            staff_info = dict()
            for staff_group in self.Parameters.staff_groups:
                staff_info.update(self.make_staff_info(self.get_staff_info(str(staff_group))))
            self.save_staff_info(staff_info)

        with self.memoize_stage.init_packages:
            self.get_versions()

        if self.Parameters.use_z2:
            with self.memoize_stage.get_versions_from_z2:
                self.get_versions_from_z2()

        with self.memoize_stage.build_image:
            self.show_versions()
            self.build_image()

        with self.memoize_stage.wait_build(commit_on_entrance=False, commit_on_wait=False):
            self.wait_task()
            self.Context.build_tasks = None
            self.Context.save()

        self.show_packer_info()
