import requests
import time


def memoize(f):
    memo = {}

    def helper(*args):
        key = '.'.join([str(i) for i in args[1:]])
        if key not in memo:
            memo[key] = f(*args)
        return memo[key]

    return helper


class PingdomCredentials(object):
    def __init__(self, login, password, app_key):
        self.login = login
        self.password = password
        self.app_key = app_key


class Pingdom(object):
    endpoint = "https://api.pingdom.com/api/2.1/"

    def __init__(self, credentials):
        """

        :type credentials: PingdomCredentials
        """
        self.creds = credentials
        self.checks = {}
        for check in self._list_checks()['checks']:
            if check['status'] != "paused":
                self.checks[check['name']] = check['id']

    def _list_checks(self):
        path = 'checks'
        return self._make_request('GET', path)

    @memoize
    def summary_average(self, check, start=None, end=None):
        path = "summary.average/{}".format(self.checks[check])

        params = {
            'includeuptime': "true"
        }

        month_seconds = 30 * 24 * 60 * 60

        if not start:
            start = time.time() - month_seconds

        params['from'] = start

        if end:
            params['to'] = end

        return self._make_request('GET', path, params=params)

    def summary_performance(self, check):
        path = "summary.performance/{}".format(self.checks[check])

        month_seconds = 30 * 24 * 60 * 60
        params = {
            'from': time.time() - month_seconds,
            'resolution': 'week',
            'includeuptime': "true"
        }
        return self._make_request('GET', path, params=params)

    def secs_down(self, check, start=None, end=None):
        summary = self.summary_average(check, start, end)['summary']
        return summary['status']['totaldown']

    def mins_down(self, check, start=None, end=None):
        return self.secs_down(check, start, end) / 60.

    def check_uptime(self, check, start=None, end=None):
        summary = self.summary_average(check, start, end)['summary']
        try:
            uptime = 100 - 100 * float(summary['status']['totaldown']) / \
                (summary['status']['totalup'] + summary['status']['totaldown'] + summary['status']['totalunknown'])
        except:
            print(summary, check)
            raise
        return round(uptime, 3)

    def _make_request(self, method, path, **kwargs):
        headers = {
            "App-Key": self.creds.app_key
        }
        http_auth = (self.creds.login, self.creds.password)

        try:

            res = requests.request(method,
                                   self.endpoint + path,
                                   headers=headers,
                                   auth=http_auth,
                                   **kwargs)
            res.raise_for_status()
        except Exception as ex:
            print(ex)
            raise
        return res.json()
