# -*- coding: utf-8 -*-

import logging
import os
import re
import subprocess as sp
import shutil
from os.path import join as pj
from collections import defaultdict

from sandbox import sdk2
import sandbox.common.types.misc as ctm
from sandbox.projects.common.BaseCompareYaMakeOutputsTask import BaseCompareYaMakeOutputsTask
from sandbox.projects.kwyt.resources import KwytDiff


TEMPLATE = '''<html>
<head>
    <link type="text/css" href="//yandex.st/bootstrap/3.3.6/css/bootstrap.min.css">
</head>
<body>
    <div class="containter">
        <div class="col-md-4">
            {link}
            <pre>{diff_description}</pre>
        </div>
    </div>
</body>
</html>
'''

SUPPORTED_TABLE_TYPES = {
    'kwyt__acks__hosts__': 'AckHosts',
    'kwyt__acks__pages__': 'AckPages',
    '^.*attrubutes$': 'Attributes',
    'kwyt__hosts__': 'Hosts',
    'kwyt__pages__': 'Pages',
    'kwyt__sitemaps__': 'Sitemaps',
    'kwyt__stats__': 'Stats'
}


class KwytCompareIntegrationTestOutput(BaseCompareYaMakeOutputsTask):
    '''
    Сравнение двух выхлопов интеграционного теста Квита
    '''
    DIFF_POSTFIX = '.diff'
    KWYT_DIFF_DIR = 'kwyt_diff'
    DIFF_TOOL_PATH = 'diff_tools/dump_diff'
    TABLE_DUMPS_DIR = 'table_dumps'

    left_build_output = None
    right_build_output = None
    testing_out_stuff_dir = None
    diff_tool_path = None
    diff_description = defaultdict(list)

    def get_pair_of_path(self, path):
        return [
            pj(side, path) if os.path.exists(pj(side, path)) else None
            for side in (self.left_build_output, self.right_build_output)
        ]

    @classmethod
    def diff_on_sets(cls, left_set, right_set):
        if left_set == right_set:
            return []

        output = set()

        for item in sorted(left_set - right_set):
            output.add('-{}'.format(item))

        for item in sorted(right_set - left_set):
            output.add('+{}'.format(item))

        return sorted(list(output))

    @staticmethod
    def get_files_relative_paths(directory):
        if directory is None:
            return
        directory_absolute_path = os.path.abspath(directory)
        for dir_path, _, filenames in os.walk(directory_absolute_path):
            for filename in filenames:
                yield pj(dir_path[len(directory_absolute_path):].strip('/'), filename)

    @staticmethod
    def get_table_type(encoded_filename, filename):
        for k, v in SUPPORTED_TABLE_TYPES.items():
            if re.match(k, encoded_filename) is not None:
                return v
        raise RuntimeError("Unsupported table {0}".format(filename))

    def compare_table_dumps(self):
        '''
        Ожидается, что дампы в self.TABLE_DUMPS_DIR разложены по каталогам относительно
        mr_prefix и записаны в yson-формате.
        Утилита сравнения требует на вход тип таблицы, который выводится из пути к файлу дампа.
        Например, для дампа hosts/delta.0 будет выведен тип таблицы Hosts
        '''
        tables_dir = pj(self.testing_out_stuff_dir, self.TABLE_DUMPS_DIR)

        left_tables, right_tables = [
            set(self.get_files_relative_paths(table_dumps_dir))
            for table_dumps_dir in self.get_pair_of_path(tables_dir)]

        self.diff_description['Tables set changes'] = self.diff_on_sets(left_tables, right_tables)

        for file_relative_path in sorted(left_tables & right_tables):
            if os.path.dirname(file_relative_path) == '':
                continue

            encoded_filename = file_relative_path.replace('/', '__')
            table_type = self.get_table_type(encoded_filename, file_relative_path)
            output_path = pj(self.KWYT_DIFF_DIR, encoded_filename + self.DIFF_POSTFIX + '.html')
            with open(output_path, 'w') as output_file:
                files = self.get_pair_of_path(pj(tables_dir, file_relative_path))
                diff_cmd_list = (
                    [
                        pj(self.right_build_output, self.testing_out_stuff_dir, self.DIFF_TOOL_PATH),
                        '--type', table_type, '--file1', files[0], '--file2', files[1]
                    ]
                )
                logging.debug('Running:\n%s', ' '.join(diff_cmd_list))
                returncode = sp.call(
                    diff_cmd_list,
                    stdout=output_file,
                    stderr=output_file
                )

            if returncode == 0:
                os.remove(output_path)
            elif returncode == 1:
                self.diff_description['Tables differ'].append(file_relative_path)
            else:
                raise sp.CalledProcessError(
                    returncode=returncode,
                    cmd=' '.join(diff_cmd_list),
                    output=open(output_path).read())

    def compare(self, build_output1, build_output2, testing_out_stuff_dir):
        self.left_build_output = build_output1
        self.right_build_output = build_output2
        self.testing_out_stuff_dir = testing_out_stuff_dir
        self.diff_tool_path = pj(testing_out_stuff_dir, 'diff_tools', 'dump_diff')

        if os.path.exists(self.KWYT_DIFF_DIR):
            shutil.rmtree(self.KWYT_DIFF_DIR)

        os.mkdir(self.KWYT_DIFF_DIR)

        self.compare_table_dumps()

        diff_description_text = ''
        link = ''

        if os.listdir(self.KWYT_DIFF_DIR):
            diff_resource = KwytDiff(
                self,
                'KwYT integration test diff',
                self.KWYT_DIFF_DIR,
                ttl=14
            )
            diff_data = sdk2.ResourceData(diff_resource)
            diff_data.path.mkdir(0o755, parents=True, exist_ok=True)
            diff_data.ready()
            link = 'If you strive to see detailed diff follow this link <a href="{0}">{0}</a>\n'.format(
                self.server.resource[diff_resource.id][:]["http"]["proxy"])

        for diff_section, item_list in sorted(self.diff_description.iteritems()):
            if not item_list:
                continue

            diff_description_text += '{}:\n'.format(diff_section)
            for item in sorted(item_list):
                diff_description_text += '\t{}\n'.format(item)
            diff_description_text += '\n'
        if diff_description_text:
            self.Context.diff_description = self.diff_description
            return TEMPLATE.format(link=link, diff_description=diff_description_text)
        else:
            return None

    @property
    def footer(self):
        diff_description = self.Context.diff_description
        output_html = ''
        if diff_description != ctm.NotExists:
            for diff_section, item_list in diff_description.iteritems():
                if not item_list:
                    continue

                output_html += '<table><thead><tr><h4>{}:</h4><tr></thead><tbody>'.format(diff_section)
                for item in sorted(item_list):
                    output_html += '<tr><td>{}</td></tr>'.format(item)
                output_html += '</tbody></table><br>'

            if output_html:
                return output_html
