# -*- coding: utf-8 -*-

from sandbox import sdk2
import json
import logging
import urllib2
import xml.etree.ElementTree


KWYT_PHONES_MAP = {
    'zagevalo': '6534',
    'izetag': '82885',
    'lazv85': '40721',
    'antervis': '61791',
    'akhovrychev': '85541',
    'gous32': '989096500565'
}

CM_HOSTS = [
    'https://kwyt-cm.n.yandex-team.ru',
    'https://superdups.n.yandex-team.ru',
    'https://filter-trie.n.yandex-team.ru',
    'https://rthub-test-control.n.yandex-team.ru',
    'https://rthub-test.n.yandex-team.ru'
]

DEFAULT_DUTY = '--Auto--'
GROUP_KWYT = 'kwyt'


class KwytAlertsManage(sdk2.Task):
    '''
    Управление нотификациями для дежурных по KwYT и RTHub
    '''
    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String('Duty login', description='Custom duty login (leave it empty to get the duty from the calendar)', multiline=False, required=True, default=DEFAULT_DUTY) as Duty:
            Duty.values[DEFAULT_DUTY] = DEFAULT_DUTY
            for k in KWYT_PHONES_MAP.keys():
                Duty.values[k] = k
        DryRun = sdk2.parameters.Bool('Dry run')
        with sdk2.parameters.String('Duty group', multiline=False, required=True, default=GROUP_KWYT) as DutyGroup:
            DutyGroup.values[GROUP_KWYT] = GROUP_KWYT

    def get_url(self, url, retries=2):
        for i in range(retries):
            try:
                req = urllib2.urlopen(url).read()
                return req
            except Exception as e:
                self.set_info('error on getting url %s: %s' % (url, e))
        raise Exception('Cannot get url %s' % url)

    @staticmethod
    def get_secret(secret_name):
        return sdk2.Vault.data('KWYT', secret_name)

    def get_duty_login(self, secret_name):
        calend_private_token = self.get_secret(secret_name)
        url = 'https://calendar.yandex-team.ru/export/rss.xml?private_token=' + calend_private_token + '&limit=1'
        resp = urllib2.urlopen(url).read()
        xp = xml.etree.ElementTree.fromstring(resp)
        entry = [el for el in xp if el.tag.endswith('entry')][0]
        title = [el for el in entry if el.tag.endswith('title')][0]
        ttext = title.text
        logging.info('calendar entry title: ' + ttext)
        for login in KWYT_PHONES_MAP.keys():
            if login in ttext:
                logging.info('duty is %s' % login)
                self.set_info('Current duty is %s@' % login)
                return login
        raise Exception('duty not found')

    def do_rest_request(self, url, data, method, token, dry_run=False):
        headers = {'Authorization': token if token.startswith('OAuth ') else ('OAuth ' + token),
                   'Content-Type': 'application/json'}
        logging.info('sending %s to %s' % (data, url))
        opener = urllib2.build_opener(urllib2.HTTPSHandler)
        req = urllib2.Request(url, data=json.dumps(data), headers=headers)
        req.get_method = lambda: method
        if dry_run:
            self.set_info('dry run, stopped before actual REST call')
            return (True, None)
        try:
            resp = opener.open(req)
            ans = resp.read()
            logging.info('Ok: %s' % ans)
            return (True, ans)
        except urllib2.HTTPError as e:
            ans = e.read()
            self.set_info('Error: ' + str(e.code) + ' ' + ans)
            return (False, ans)
        except Exception as e:
            self.set_info('Error: %s' % e)
            return (False, None)

    def set_phone_redirect(self, recipient_phone, robot_phone, secret_name):
        token = self.get_secret(secret_name)
        url = 'https://telegraph.yandex-team.ru/api/v3/cucm/translation_pattern/pattern/' + robot_phone
        # 98 - префикс внешнего номера
        # 55 - префикс форварда на мобильный номер с рабочего телефона
        # см. https://wiki.yandex-team.ru/diy/office/phone/
        if not recipient_phone.startswith('98'):
            recipient_phone = '55' + recipient_phone
        data = {'calledPartyTransformationMask': recipient_phone}
        return self.do_rest_request(url, data, 'PUT', token, self.Parameters.DryRun)[0]

    def set_cm_host_sms_login(self, host, recipient_login):
        url = host + '/variables/set?name=SMSTO&master=&value=' + recipient_login
        logging.info('sending command to %s' % url)
        if self.Parameters.DryRun:
            self.set_info('dry run, stopped before CM SMSTO var setting')
            return True
        try:
            resp = urllib2.urlopen(url)
            ans = resp.read()
            logging.info('Ok: %s' % ans)
            return True
        except urllib2.HTTPError as e:
            ans = e.read()
            self.set_info('Error: ' + str(e.code) + ' ' + ans)
            return False
        except Exception as e:
            self.set_info('Error: %s' % e)
            return False

    def set_cm_sms_login(self, recipient_login):
        for host in CM_HOSTS:
            success = self.set_cm_host_sms_login(host, recipient_login)
            if not success:
                return False
        return True

    def set_juggler_sms_login(self, recipient_login, secret_name):
        token = self.get_secret(secret_name)
        JUGGLER_API_HOST = 'https://juggler-api.search.yandex.net'
        url = JUGGLER_API_HOST + '/api/notify_rules/get_notify_rules?do=1'
        data = {
            'filters': [
                {'rule_id': '5aec6c49de3a6a0096da9008'}
            ]
        }
        (success, text) = self.do_rest_request(url, data, 'POST', token, False)
        if not success:
            return False

        obj = json.loads(text)
        data = obj['rules'][0]
        data.pop('creation_time', '')
        data.pop('hints', '')
        data['template_kwargs']['login'] = recipient_login
        url = JUGGLER_API_HOST + '/api/notify_rules/add_or_update_notify_rule?do=1'
        return self.do_rest_request(url, data, 'POST', token, self.Parameters.DryRun)[0]

    def on_execute(self):
        if self.Parameters.DutyGroup == GROUP_KWYT:
            ROBOT_PHONE = '26211'
            TOKEN_CALENDAR_NAME = 'calendar_kwyt_duty'
            TOKEN_TELEGRAPH_NAME = 'telegraph_kwyt_duty'
            TOKEN_JUGGLER_NAME = 'juggler_kwyt_duty'
            self.set_info('will set KwYT/RTHub duty')
        else:
            raise Exception('internal error, unknown group %s' % self.Parameters.DutyGroup)

        if not int(ROBOT_PHONE):
            raise Exception('internal error, robot phone incorrect')

        new_duty = self.Parameters.Duty
        if new_duty == DEFAULT_DUTY:
            logging.info('getting duties from calendar')
            new_duty = self.get_duty_login(TOKEN_CALENDAR_NAME)
        try:
            new_duty_phone = KWYT_PHONES_MAP[new_duty]
        except:
            raise Exception('incorrect new duty phone')
        success = self.set_phone_redirect(new_duty_phone, ROBOT_PHONE, TOKEN_TELEGRAPH_NAME)
        if not success:
            raise Exception('failed to set the phone redirect')
        success = self.set_cm_sms_login(new_duty)
        if not success:
            raise Exception('failed to set the CM SMSTO variable')
        success = self.set_juggler_sms_login(new_duty, TOKEN_JUGGLER_NAME)
        if not success:
            raise Exception('failed to set the Juggler SMS notification login')
