from sandbox import sdk2
from sandbox.common.types import task
import sandbox.projects.common.constants as consts
import sandbox.projects.kwyt.resources as res
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from os.path import join as pj

import logging
import os
import subprocess


_SELECTQL_BIN_PATH = 'robot/kwyt/tools/selectql'


class SelectqlTask(sdk2.Task):
    """
    Start selectql task.
    """

    class Requirements(sdk2.Task.Requirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Svn url for arcadia (you can add '@<commit_number>')",
            required=True,
            default_value='arcadia:/arc/trunk/arcadia'
        )
        selectqlBinResource = sdk2.parameters.Resource(
            "Resource id of selectql (if empty - will be built)",
            resource_type=res.SelectqlBin
        )
        yqlToken = sdk2.parameters.YavSecret("YQL token", default="", multiline=False)
        yqlTokenKey = sdk2.parameters.String("YQL token key", default="yql.token", multiline=False)
        ytToken = sdk2.parameters.YavSecret("YT token", default="", multiline=False)
        ytTokenKey = sdk2.parameters.String("YT token key", default="yt.token", multiline=False)
        pool = sdk2.parameters.String("Pool", default="", multiline=False)
        where = sdk2.parameters.String("Where", default="", multiline=False)
        urlsList = sdk2.parameters.String("Urls list", default="", multiline=True)
        columns = sdk2.parameters.String("Columns", default="", multiline=False)
        output = sdk2.parameters.String("Output table", default="", multiline=False)
        limit = sdk2.parameters.Integer("Row limit")
        getLastVersion = sdk2.parameters.Bool("Get last version of document")
        kwytPath = sdk2.parameters.String("Kwyt path", default="//home/kwyt/pages/", multiline=False)
        cluster = sdk2.parameters.String("Cluster", default="arnold", multiline=False)

    def buildSelectql(self, selectqlBinPath):
        subtask = sdk2.Task["YA_MAKE_2"](
            self,
            description="Child of {}".format(self.id),
            owner=self.owner,
            checkout_arcadia_from_url=self.Parameters.arcadia_url,
            targets=selectqlBinPath,
            arts=pj(selectqlBinPath, 'selectql'),
            result_single_file=True,
            result_rt='SELECTQL_BIN',
            result_rd='selectql binary',
            use_aapi_fuse=True,
            aapi_fallback=True,
            build_system=consts.SEMI_DISTBUILD_BUILD_SYSTEM,
            checkout_mode='auto'
        )
        subtask.enqueue()

        return subtask.id

    class Context(sdk2.Context):
        errors = []
        messages = []
        subtasks = {}

    def checkSubtaskStatus(self, subtask):
        if subtask.status in (task.Status.FAILURE, task.Status.EXCEPTION):
            self.Context.errors.append('Subtask {} failed'.format(subtask.id))
        else:
            self.Context.messages.append('Subtask {} have finished successfully'.format(subtask.id))

    def on_execute(self):
        if not self.Parameters.selectqlBinResource:
            with self.memoize_stage.build_selectql:
                subtask = self.buildSelectql(_SELECTQL_BIN_PATH)
                self.Context.subtasks['SELECTQL_BIN'] = subtask
                raise sdk2.WaitTask([subtask], task.Status.Group.FINISH | task.Status.Group.BREAK, wait_all=True)

        subtasks = list(self.find())
        if subtasks:
            for subtask in subtasks:
                self.checkSubtaskStatus(subtask)

        if self.Context.errors:
            raise SandboxTaskFailureError('One of child tasks failed unexpectedly')

        if self.Parameters.selectqlBinResource:
            selectqlBinRes = self.Parameters.selectqlBinResource
        else:
            selectqlBinRes = sdk2.Resource.find(
                type='SELECTQL_BIN',
                task_id=self.Context.subtasks['SELECTQL_BIN']
            ).first()
            logging.info("selectql resource: {}".format(selectqlBinRes.id))
        selectqlBin = str(sdk2.ResourceData(selectqlBinRes).path)
        urlsListFileName = pj(os.getcwd(), 'urlsList')
        if self.Parameters.urlsList:
            urlsListFile = open(urlsListFileName, 'w')
            urlsListFile.write(self.Parameters.urlsList)
            urlsListFile.close()

        cmd = [
            selectqlBin,
            '--columns', self.Parameters.columns,
            '--output', self.Parameters.output,
            '--yql-token', self.Parameters.yqlToken.data()[self.Parameters.yqlTokenKey],
            '--yt-token', self.Parameters.ytToken.data()[self.Parameters.ytTokenKey]
            ]
        if self.Parameters.urlsList:
            urlsListFile = open(pj(os.getcwd(), 'urlsList'), 'w')
            urlsListFile.write(self.Parameters.urlsList)
            urlsListFile.close()
            cmd.extend(['--urls-list', pj(os.getcwd(), 'urlsList')])
        if self.Parameters.where:
            cmd.extend(['--where', self.Parameters.where])
        if self.Parameters.pool:
            cmd.extend(['--pool', self.Parameters.pool])
        if self.Parameters.limit:
            cmd.extend(['--limit', self.Parameters.limit])
        if self.Parameters.getLastVersion:
            cmd.append('--get-last-version')
        if self.Parameters.kwytPath:
            cmd.extend(['--kwyt-path', self.Parameters.kwytPath])
        if self.Parameters.cluster:
            cmd.extend(['--cluster', self.Parameters.cluster])

        logging.info("Run cmd: {}".format(' '.join(cmd)))
        try:
            subprocess.check_call(cmd)
        finally:
            if os.path.exists(urlsListFileName):
                os.remove(urlsListFileName)
