# -*- coding: utf-8 -*-

import urllib
import zipfile

from sandbox import sdk2

from sandbox.projects.common.nanny import nanny

TEAM = ["dieash"]


class AnonymousIpV4Resource(sdk2.Resource):
    """
    AnonymousIP CSV file from MaxMind. IPv4 version
    """
    any_arch = True
    executable = False
    releasable = True
    releasers = TEAM


class AnonymousIpV6Resource(sdk2.Resource):
    """
    AnonymousIP CSV file from MaxMind. IPv6 version
    """
    any_arch = True
    executable = False
    releasable = True
    releasers = TEAM


RESULT_FILE_V4 = 'GeoIP2-Anonymous-IP-Blocks-IPv4.csv'
RESULT_FILE_V6 = 'GeoIP2-Anonymous-IP-Blocks-IPv6.csv'
MAXMIND_URL = 'https://download.maxmind.com/app/geoip_download?edition_id=GeoIP2-Anonymous-IP-CSV&suffix=zip&license_key=%(license_key)s'


class AnonymousIpCollectTask(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Download AnonymousIP CSV files from MaxMind
    """

    def on_execute(self):
        license_key = sdk2.Vault.data('GEOLOCATION', 'maxmind_license_key')

        url = MAXMIND_URL % {'license_key': license_key}
        zip_name = 'GeoIP2-Anonymous-IP-CSV.zip'
        urllib.urlretrieve(url, zip_name)

        with zipfile.ZipFile(zip_name, 'r') as zip:
            real_path_v4 = [name for name in zip.namelist() if name.endswith(RESULT_FILE_V4)]
            real_path_v6 = [name for name in zip.namelist() if name.endswith(RESULT_FILE_V6)]

            if len(real_path_v4) < 1:
                raise Exception("%s not found in MaxMind archive" % RESULT_FILE_V4)
            if len(real_path_v4) > 1:
                raise Exception("Multiple %s files in MaxMind archive" % RESULT_FILE_V4)
            if len(real_path_v6) < 1:
                raise Exception("%s not found in MaxMind archive" % RESULT_FILE_V6)
            if len(real_path_v6) > 1:
                raise Exception("Multiple %s files in MaxMind archive" % RESULT_FILE_V6)

            inner_path_v4 = real_path_v4[0]
            inner_path_v6 = real_path_v6[0]
            extracted_path_v4 = zip.extract(inner_path_v4)
            extracted_path_v6 = zip.extract(inner_path_v6)

        resource_v4 = sdk2.ResourceData(AnonymousIpV4Resource(
            self, "AnonymousIP CSV file from MaxMind. IPv4 version. %s" % inner_path_v4, extracted_path_v4
        ))
        resource_v4.ready()

        resource_v6 = sdk2.ResourceData(AnonymousIpV6Resource(
            self, "AnonymousIP CSV file from MaxMind. IPv6 version. %s" % inner_path_v6, extracted_path_v6
        ))
        resource_v6.ready()
