# -*- coding: utf-8 -*-

from sandbox import sdk2

import sandbox.common.errors as ce
from sandbox.common.types import client as ctc
from sandbox.projects.common.nanny import nanny
from sandbox.projects.geobase.Geodata6BinExp.resource import GEODATA6BIN_EXP
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp

import logging
import os
import re

TEAM = ["dieash"]
MAX_TEST_GEOBASES_COUNT = 5
TEST_GEOBASES_FOLDER = "test_geobases"
TEST_GEOBASE_NAME_RE = re.compile('^[A-z\d_\-\.]+$')


class LaasTestGeobasesResource(sdk2.Resource):
    """Folder %s with test geobase files""" % TEST_GEOBASES_FOLDER
    any_arch = True
    executable = False
    releasable = True
    releasers = TEAM


class LaasCollectTestGeobasesTask(nanny.ReleaseToNannyTask2, sdk2.Task):
    """Task to collect test geobases for LaaS"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL
        privileged = True

    def execute(self, command, description):
        with ProcessLog(self, logger=logging.getLogger(description)) as process_log:
            return sp.Popen(
                command,
                shell=True,
                stdout=process_log.stdout,
                stderr=process_log.stdout,
                cwd=os.getcwd()
            ).wait()

    def install_dependencies(self):
        self.execute('sudo apt-get update', 'installation')
        self.execute('sudo apt-get --assume-yes install libgeobase6-python python-ipaddr', 'installation')

    def get_test_geobases(self):
        valid_active_values = ["true", "True"]

        test_geobases = []

        for active_value in valid_active_values:
            geobases = sdk2.Resource.find(
                GEODATA6BIN_EXP,
                attrs=dict(active=active_value)
            ).limit(MAX_TEST_GEOBASES_COUNT + 1)
            test_geobases += list(geobases)

        if len(test_geobases) == 0:
            logging.info("No active test geobases found")
            return

        if len(test_geobases) > MAX_TEST_GEOBASES_COUNT:
            raise ce.TaskError("More than %d active test geobases" % MAX_TEST_GEOBASES_COUNT)

        return test_geobases

    def on_execute(self):
        self.install_dependencies()

        test_geobases = self.get_test_geobases()

        test_geobases_resource = LaasTestGeobasesResource(self, LaasTestGeobasesResource.description,
                                                          TEST_GEOBASES_FOLDER)
        test_geobases_resource_data = sdk2.ResourceData(test_geobases_resource)
        test_geobases_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)

        for test_geobase in test_geobases:
            self.check_geobase(test_geobase)

            name = test_geobase.name

            logging.info("Copying geobase name='%s' id=%d" % (name, test_geobase.id))
            copy_path(str(sdk2.ResourceData(test_geobase).path), str(test_geobases_resource_data.path.joinpath(name)))

        test_geobases_resource_data.ready()

    def check_geobase(self, test_geobase):
        try:
            name = test_geobase.name
        except AttributeError:
            raise ce.TaskError("Geobase with id=%d does not have a name" % test_geobase.id)

        if not TEST_GEOBASE_NAME_RE.match(name):
            raise ce.TaskError("Name of the geobase with id=%d does not match regular expression. "
                               "Name = '%s'" % (test_geobase.id, name))

        logging.info("Checking geobase name='%s' id=%d" % (name, test_geobase.id))

        test_geobase_path = str(sdk2.ResourceData(test_geobase).path)

        checker_paths = [
            "laas/CollectTestGeobases/check_geobase.py",
            "geobase/common/ipreg-testing-v6.py",
            "geobase/common/isp-testing.py",
            "geobase/common/maxmind_anon_data_processor.py",
            "geobase/common/maxmind_isp_data_processor.py",
        ]

        for checker_path in checker_paths:
            absolute_path = os.path.join(os.path.dirname(__file__), "..", "..", checker_path)
            description = "[check %s]" % name
            command = "python %s --geodata '%s'" % (absolute_path, test_geobase_path)

            status = self.execute(command, description)

            if status != 0:
                message = "command[%s] => failed with %s" % (command, status)
                raise ce.TaskError(message)
